package de.upb.pga3.panda2.client.gui;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Optional;

import de.upb.pga3.panda2.client.core.AnalysisRegistry;
import de.upb.pga3.panda2.client.core.ResultLoader;
import de.upb.pga3.panda2.client.core.ResultStorer;
import de.upb.pga3.panda2.core.datastructures.AnalysisResult;
import javafx.application.Platform;
import javafx.scene.control.Alert;
import javafx.scene.control.Alert.AlertType;
import javafx.scene.control.ButtonType;
import javafx.scene.web.WebEngine;
import javafx.scene.web.WebView;
import javafx.stage.FileChooser;
import javafx.stage.Modality;
import javafx.stage.Stage;
import netscape.javascript.JSObject;

/**
 * Controller class of the GUI.
 *
 * @author Felix
 */
public class Controller {
	// MVC
	GUIView view;
	Model model;

	private final FileChooser chooserDialog;

	Controller(final GUIView view, final Model model) {
		this.model = model;
		this.view = view;

		this.chooserDialog = new FileChooser();
		final FileChooser.ExtensionFilter apkFilter = new FileChooser.ExtensionFilter(
				"*.pa2 - Panda2 Analysis Result File", "*.pa2");
		final FileChooser.ExtensionFilter allFilesFilter = new FileChooser.ExtensionFilter("*.* - All files", "*.*");
		this.chooserDialog.getExtensionFilters().addAll(apkFilter, allFilesFilter);
	}

	Model getModel() {
		return this.model;
	}

	GUIView getView() {
		return this.view;
	}

	public void newClicked() {
		if (Wizard.getInstance() == null) {
			final Stage subStage = new Stage();
			subStage.initOwner(getView().stage);
			subStage.initModality(Modality.WINDOW_MODAL);
			new Wizard(getModel()).start(subStage);
		} else {
			Wizard.getInstance().show();
		}
	}

	public void openClicked() {
		this.chooserDialog.setInitialDirectory(GuiConfig.getInstance().getOpenFolder());
		final File file = this.chooserDialog.showOpenDialog(getView().stage);
		if (file != null && file.getParentFile().exists()) {
			GuiConfig.getInstance().setOpenFolder(file.getParentFile());
			final AnalysisResult loadedResult = ResultLoader.loadPreviousAnalysisResult(file.toString());

			getModel().lastSetup = new ArrayList<>();
			getModel().lastSetup.add(AnalysisRegistry.getInstance().getName(loadedResult));
			getModel().loadedCounter++;
			getModel().lastSetup.add("Loaded previous result (" + getModel().loadedCounter + ")");

			getModel().mapAnalysisResults.put(loadedResult, getModel().lastSetup);
			GUIView.log("Analysis result loaded");
			getView().loadAnalysisResult(loadedResult);
		}
	}

	public void saveClicked() {
		if (getModel().saveFile != null) {
			save();
		} else {
			saveAsClicked();
		}
	}

	public void saveAsClicked() {
		this.chooserDialog.setInitialDirectory(GuiConfig.getInstance().getSaveFolder());

		getModel().saveFile = this.chooserDialog.showSaveDialog(getView().stage);
		if (getModel().saveFile != null) {
			GuiConfig.getInstance().setSaveFolder(getModel().saveFile.getParentFile());
			save();
		}
	}

	private void save() {
		if (getModel().saveFile != null) {
			if (getModel().saveFile.exists()) {
				final Alert alert = new Alert(AlertType.CONFIRMATION);
				final Stage alertStage = (Stage) alert.getDialogPane().getScene().getWindow();
				GUIHelper.setPandaIcon(alertStage);
				alert.setTitle("Save");
				alert.setHeaderText("The file\n" + getModel().saveFile.getAbsolutePath() + "\nalready exists.");
				alert.setContentText("Overwrite it?");

				final Optional<ButtonType> result = alert.showAndWait();
				if (result.get() == ButtonType.OK) {
					ResultStorer.storeResult(getModel().currentAnalysisResult, getModel().saveFile.toString());
					getView().log("Result saved in file: " + getModel().saveFile);
				} else {
					alert.hide();
					getView().stage.show();
				}
			} else {
				ResultStorer.storeResult(getModel().currentAnalysisResult, getModel().saveFile.toString());
				getView().log("Result saved in file: " + getModel().saveFile);
			}
		}
	}

	public void exitClicked() {
		final Alert alert = new Alert(AlertType.CONFIRMATION);
		final Stage alertStage = (Stage) alert.getDialogPane().getScene().getWindow();
		GUIHelper.setPandaIcon(alertStage);
		alert.setTitle("Exit");
		alert.setHeaderText("You will exit now.\nResults which are not saved will be discarded.");
		alert.setContentText("Proceed?");

		final Optional<ButtonType> result = alert.showAndWait();
		if (result.get() == ButtonType.OK) {
			Platform.exit();
		} else {
			alert.hide();
			getView().stage.show();
		}
	}

	public boolean removeFilter(final String filter) {
		final Alert alert = new Alert(AlertType.CONFIRMATION);
		final Stage alertStage = (Stage) alert.getDialogPane().getScene().getWindow();
		GUIHelper.setPandaIcon(alertStage);
		alert.setTitle("Add filter");
		alert.setHeaderText(filter + " filter is already active.");
		alert.setContentText("Do you want to remove it instead?");

		final Optional<ButtonType> result = alert.showAndWait();
		if (result.get() == ButtonType.OK) {
			return true;
		} else {
			alert.hide();
			getView().stage.show();
			return false;
		}
	}

	private static final String JS_DEV_PX_RATIO = "devicePixelRatio";
	private static final String JS_WINDOW = "window";

	static double getDevPxRatio(final WebEngine engine) {
		final JSObject window = (JSObject) engine.executeScript(JS_WINDOW);
		return ((Number) window.getMember(JS_DEV_PX_RATIO)).doubleValue();
	}

	void zoomIn() {
		if (getView().tabPane.getSelectionModel().getSelectedItem() == getView().tabTextual) {
			getModel().zoomTextual += 0.1d;
			setZoom(getView().htmlTextual, getModel().initDevPxRatioTextual, getModel().zoomTextual);
		} else if (getView().tabPane.getSelectionModel().getSelectedItem() == getView().tabGraphical) {
			getModel().zoomGrapical += 0.1d;
			setZoom(getView().htmlGraphical, getModel().initDevPxRatioGraphical, getModel().zoomGrapical);
		}
	}

	void zoomOut() {
		if (getView().tabPane.getSelectionModel().getSelectedItem() == getView().tabTextual) {
			getModel().zoomTextual -= 0.1d;
			setZoom(getView().htmlTextual, getModel().initDevPxRatioTextual, getModel().zoomTextual);
		} else if (getView().tabPane.getSelectionModel().getSelectedItem() == getView().tabGraphical) {
			getModel().zoomGrapical -= 0.1d;
			setZoom(getView().htmlGraphical, getModel().initDevPxRatioGraphical, getModel().zoomGrapical);
		}
	}

	void zoomReset() {
		if (getView().tabPane.getSelectionModel().getSelectedItem() == getView().tabTextual) {
			getModel().zoomTextual = 1.0d;
			setZoom(getView().htmlTextual, getModel().initDevPxRatioTextual, getModel().zoomTextual);
		} else if (getView().tabPane.getSelectionModel().getSelectedItem() == getView().tabGraphical) {
			getModel().zoomGrapical = 1.0d;
			setZoom(getView().htmlGraphical, getModel().initDevPxRatioGraphical, getModel().zoomGrapical);
		}
	}

	private static void setZoom(final WebView view, final double initPxRatio, final double zoom) {
		final JSObject window = (JSObject) view.getEngine().executeScript(JS_WINDOW);
		window.setMember(JS_DEV_PX_RATIO, initPxRatio * zoom);
		view.setZoom(zoom);
	}

	public void loadMessages(final int current) {
		getView().showNextMessages(current);
	}

	public void viewInBrowser(final boolean generateFileFirst) {
		String content;
		if (getView().tabPane.getSelectionModel().getSelectedItem() == getView().tabTextual) {
			content = getView().htmlTextualStr;
		} else {
			content = getView().htmlGraphicalStr;
		}
		if (generateFileFirst) {
			try {
				final Path tmp = Files.createTempFile("tmpResult", ".html");
				Files.write(tmp, content.getBytes(), StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING,
						StandardOpenOption.WRITE);
				content = tmp.toString();
			} catch (final IOException e) {
				errorOpeningInBrowser();
				GUIView.LOGGER.debug("Write tmp file: {}", e.getMessage());
				return;
			}
		}
		getView().getHostServices().showDocument(content);
	}

	private void errorOpeningInBrowser() {
		final Alert alert = new Alert(AlertType.INFORMATION);
		final Stage alertStage = (Stage) alert.getDialogPane().getScene().getWindow();
		GUIHelper.setPandaIcon(alertStage);
		alert.setTitle("Error");
		alert.setHeaderText("The Browser could not be accessed.");

		alert.showAndWait();
		alert.hide();
		getView().stage.show();
	}
}