package de.upb.pga3.panda2.core.datastructures;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import gnu.trove.strategy.HashingStrategy;
import gnu.trove.strategy.IdentityHashingStrategy;
import soot.toolkits.graph.DirectedGraph;
import soot.util.HashMultiMap;

/**
 * AnalysisGraph is mainly used for generating a graph based on the Input. It
 * implements the interface GraphGenerator
 *
 * @author nptsy
 * @author Fabian
 * @author RamKumar
 */
public class AnalysisGraph implements DirectedGraph<Object> {
	private static final Logger LOGGER = LogManager.getLogger(AnalysisGraph.class);

	// list of message for the result
	private final List<Message> mLstMessage;

	// all outgoing transitions for nodes
	protected transient final HashMultiMap<Object, Transition> mMapTransitionOut;
	// all ingoing transitions for nodes
	protected transient final HashMultiMap<Object, Transition> mMapTransitionIn;

	// the input that contains all information about objects in android app
	private final Input mInput;

	/**
	 * Constructor using the {@link IdentityHashingStrategy} for mapping
	 * transitions to nodes
	 *
	 * @param inInput
	 *            the EnhancedInput that contains all information abt objects in
	 *            android app
	 */
	public AnalysisGraph(final Input inInput) {
		this(inInput, new IdentityHashingStrategy<>(), new IdentityHashingStrategy<Transition>());
	}

	/**
	 * Constructor
	 *
	 * @param inInput
	 *            the EnhancedInput that contains all information abt objects in
	 *            android app
	 */
	protected AnalysisGraph(final Input inInput, final HashingStrategy<Object> keyStrategy,
			final HashingStrategy<Transition> valueStrategy) {

		this.mInput = inInput;
		// initialize data structures
		this.mMapTransitionOut = new HashMultiMap<>();
		this.mMapTransitionIn = new HashMultiMap<>();
		this.mLstMessage = new ArrayList<>();
	}

	/**
	 * Add a new transition to current the graph
	 *
	 * @param inTransition
	 *            The transition
	 * @return {@code true} if and only if the transition was successfully added
	 */
	public boolean addTransition(final Transition inTransition) {

		boolean added = true;
		added = added && this.mMapTransitionOut.put(inTransition.getSource(), inTransition);
		added = added && this.mMapTransitionIn.put(inTransition.getTarget(), inTransition);
		if (!added) {
			LOGGER.trace("Transition from '{}' to '{}' could not be added", inTransition.mEleFrom.toString(),
					inTransition.mEleTo.toString());
		}
		return added;
	}

	/**
	 * Remove an existing transition from the current graph
	 *
	 * @param inTransition
	 *            The transition
	 * @return {@code true} if and only if the transition was successfully
	 *         removed
	 */
	public boolean removeTransition(final Transition inTransition) {

		boolean removed = true;
		removed = removed && this.mMapTransitionOut.remove(inTransition.getSource(), inTransition);
		removed = removed && this.mMapTransitionIn.remove(inTransition.getTarget(), inTransition);
		if (!removed) {
			LOGGER.trace("Transition from '{}' to '{}' could not be removed", inTransition.mEleFrom.toString(),
					inTransition.mEleTo.toString());
		}
		return removed;
	}

	/**
	 * get outgoing transition of an element
	 *
	 * @param inElement
	 *            that outgoing transition must be got
	 * @return list of transitions that are outgoing transitions of the input
	 *         element
	 */
	public Set<Transition> getOutgoingTransitions(final Object inElement) {

		if (this.mMapTransitionOut.containsKey(inElement)) {
			return this.mMapTransitionOut.get(inElement);
		} else {
			return Collections.emptySet();
		}
	}

	/**
	 * Return list of all Transitions added to the Graph
	 *
	 * @return list of transitions
	 *
	 */
	@Deprecated
	public List<Transition> getTransitionList() {
		return new ArrayList<>(this.mMapTransitionIn.values());
	}

	/**
	 * get incoming transition of an element
	 *
	 * @param inElement
	 *            that incoming transition must be got
	 * @return list of transitions that are incoming transitions of the input
	 *         element
	 */

	public Set<Transition> getIncomingTransitions(final Object inElement) {

		if (this.mMapTransitionIn.containsKey(inElement)) {
			return this.mMapTransitionIn.get(inElement);
		} else {
			return Collections.emptySet();
		}
	}

	/**
	 * Returns the set of nodes in that graph. A node is defined as an object
	 * (element) with incoming or outgoing transitions.
	 *
	 * @return The set of nodes in that graph
	 */
	public Set<Object> getNodes() {

		final Set<Object> nodes = new HashSet<>(this.mMapTransitionIn.keySet());
		nodes.addAll(this.mMapTransitionOut.keySet());
		return nodes;
	}

	/**
	 * add message for the current result of graph
	 *
	 * @param inMsg
	 *            message
	 */
	public void addMessage(final Message inMsg) {
		this.mLstMessage.add(inMsg);
	}

	/**
	 * get list of messages for current result of graph
	 *
	 * @return list of message
	 */
	public List<Message> getMessages() {
		return this.mLstMessage;
	}

	/**
	 * get input of current graph
	 *
	 * @return the input of current graph
	 */
	public Input getInput() {
		return this.mInput;
	}

	@Override
	public List<Object> getHeads() {
		throw new UnsupportedOperationException();
	}

	@Override
	public List<Object> getTails() {
		throw new UnsupportedOperationException();
	}

	@Override
	public List<Object> getPredsOf(final Object s) {

		final Set<Transition> ingoing = this.mMapTransitionIn.get(s);
		if (ingoing != null) {
			return ingoing.stream().map(t -> t.getSource()).collect(Collectors.toList());
		} else {
			return new ArrayList<>(0);
		}
	}

	@Override
	public List<Object> getSuccsOf(final Object s) {

		final Set<Transition> outgoing = this.mMapTransitionOut.get(s);
		if (outgoing != null) {
			return outgoing.stream().map(t -> t.getTarget()).collect(Collectors.toList());
		} else {
			return new ArrayList<>(0);
		}
	}

	@Override
	public int size() {
		return getNodes().size();
	}

	@Override
	public Iterator<Object> iterator() {
		return getNodes().iterator();
	}

}
