package de.upb.pga3.panda2.core.datastructures;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import de.upb.pga3.panda2.client.core.datastructures.ManifestInfo;
import de.upb.pga3.panda2.core.services.IntentInformation;
import de.upb.pga3.panda2.extension.Enhancer;
import soot.Body;
import soot.Scene;
import soot.SootClass;
import soot.SootField;
import soot.SootMethod;
import soot.Unit;
import soot.util.HashMultiMap;
import soot.util.MultiMap;

/**
 * The {@link Enhancer} will create the {@link EnhancedInput}. It will contain
 * all the information about the App's source code and the permissions used by
 * statements. As well as it provides some methods to navigate through the
 * representation of the App's source code.
 *
 * @author Fabian
 * @author Felix
 *
 */
public class EnhancedInput implements Input {

	private transient final Map<Unit, Body> unitToBodyMap;
	private final Map<String, String> unitToPkgStrMap;
	private transient final MultiMap<Object, Permission> permissionMap;
	private transient final MultiMap<SootClass, String> androidComponents;
	private final List<String> androidComponentsByName;
	private transient final List<Object> maybeMore;
	private transient Map<String, Collection<IntentInformation>> intentMap;

	private transient Scene scene;

	private final String appName;
	private ManifestInfo appManifestInfo;

	public EnhancedInput(final String appName, final ManifestInfo appManifestInfo) {

		this.unitToBodyMap = new HashMap<>();
		this.unitToPkgStrMap = new HashMap<>();
		this.permissionMap = new HashMultiMap<>();
		this.androidComponents = new HashMultiMap<>();
		this.androidComponentsByName = new ArrayList<>();
		this.maybeMore = new ArrayList<>();
		this.intentMap = new HashMap<>();
		this.appName = appName;
		this.appManifestInfo = appManifestInfo;
	}

	public void setScene(final Scene scene) {
		this.scene = scene;
	}

	public void mapUnitToBody(final Unit u, final Body b) {
		this.unitToBodyMap.put(u, b);
	}

	public Body getBodyForUnit(final Unit u) {
		return this.unitToBodyMap.get(u);
	}

	public void mapUnitByStrToPkgStr(final String uStr, final String pkgStr) {
		this.unitToPkgStrMap.put(uStr, pkgStr);
	}

	public String getPkgStrForUnit(final Unit u) {
		return this.unitToPkgStrMap.get(u);
	}

	public void addPermission(final Permission p) {
		addPermissionTo(this, p);
	}

	public void addPermissionTo(final Object codeObject, final Permission p) {
		if (codeObject instanceof Body) {
			this.permissionMap.put(((Body) codeObject).getMethod(), p);
		} else {
			this.permissionMap.put(codeObject, p);
		}
	}

	public void setAsAndroidComponent(final SootClass c) {
		setAsAndroidComponent(c, new HashSet<String>());
	}

	public void setAsAndroidComponent(final SootClass c, final Set<String> contentProviderURLs) {
		if (this.scene.containsClass(c.getName())) {
			if (contentProviderURLs == null || contentProviderURLs.isEmpty()) {
				this.androidComponents.put(c, null);
			} else {
				this.androidComponents.putAll(c, contentProviderURLs);
			}
			this.androidComponentsByName.add(c.getName());
		} else {
			throw new IllegalArgumentException("The given class is not part of the Android App!");
		}
	}

	public boolean isAndroidComponent(final SootClass c) {
		return this.androidComponents.containsKey(c);
	}

	public boolean isAndroidComponent(final String className) {
		return this.androidComponentsByName.contains(className);
	}

	public Collection<SootClass> getAndroidComponents() {
		return Collections.unmodifiableCollection(this.androidComponents.keySet());
	}

	public String getAppName() {
		return this.appName;
	}

	public Collection<Permission> getPermissions(final Object obj) {
		return this.permissionMap.get(obj);
	}

	public Collection<Permission> getPermissionsFor(final Unit u) {
		return this.permissionMap.get(u);
	}

	public Collection<Permission> getPermissionsFor(final SootMethod m) {
		return this.permissionMap.get(m);
	}

	public Collection<Permission> getPermissionsFor(final Body b) {
		return this.permissionMap.get(b.getMethod());
	}

	public Collection<Permission> getPermissionsFor(final SootField f) {
		return this.permissionMap.get(f);
	}

	public Collection<Permission> getPermissionsFor(final SootClass c) {
		return this.permissionMap.get(c);
	}

	public Collection<Permission> getPermissions() {
		return this.permissionMap.get(this);
	}

	public SootClass[] getAppClasses() {
		final Collection<SootClass> sootClasses = this.scene.getApplicationClasses();
		return sootClasses.toArray(new SootClass[0]);
	}

	public SootMethod[] getAppEntryPoints() {
		return this.scene.getEntryPoints().toArray(new SootMethod[0]);
	}

	public List<Object> getChildren(final Object obj) {
		if (obj instanceof Unit) {
			return null;
		} else {
			if (obj instanceof SootMethod) {
				final List<Object> tempLst = new ArrayList<>();
				if (!((SootMethod) obj).getDeclaringClass().isInterface()) {
					try {
						tempLst.addAll(((SootMethod) obj).retrieveActiveBody().getUnits());
						return tempLst;
					} catch (final RuntimeException e) {
						if (e.getMessage().contains("No method source set for method")) {
							// Message was already given by the Enhancer. Do
							// nothing.
						} else {
							e.printStackTrace();
						}
						return null;
					}
				} else {
					return null;
				}
			} else if (obj instanceof Body) {
				final List<Object> tempLst = new ArrayList<>();
				tempLst.addAll(((Body) obj).getUnits());
				return tempLst;
			} else if (obj instanceof SootClass) {
				final List<Object> tempLst = new ArrayList<>();
				tempLst.addAll(((SootClass) obj).getMethods());
				return tempLst;
			} else if (obj instanceof EnhancedInput) {
				return new ArrayList<>(Arrays.asList(getAppClasses()));
			} else {
				return null;
			}
		}
	}

	public SootClass getComponentByName(final String inComponentName) {
		for (final SootClass component : getAndroidComponents()) {
			if (component.getName().equals(inComponentName)) {
				return component;
			}
		}
		return null;
	}

	public void addToMaybeMoreList(final Object item) {
		this.maybeMore.add(item);
	}

	public List<Object> getMaybeMoreList() {
		return this.maybeMore;
	}

	public Map<String, Collection<IntentInformation>> getIntents() {
		return this.intentMap;
	}

	public Collection<IntentInformation> getIntent(final String classNameOrActionString) {
		return this.intentMap.get(classNameOrActionString);
	}

	public void addIntent(final IntentInformation inIntent) {
		final String classNameOrActionString = inIntent.getClassNameOrActionString();
		Collection<IntentInformation> currentList = this.intentMap.get(classNameOrActionString);
		this.intentMap.remove(classNameOrActionString);
		if (currentList == null) {
			currentList = new ArrayList<>();
		}
		currentList.add(inIntent);
		this.intentMap.put(classNameOrActionString, currentList);
	}

	public void addIntents(final Collection<IntentInformation> inIntents) {
		for (final IntentInformation intent : inIntents) {
			final String classNameOrActionString = intent.getClassNameOrActionString();
			Collection<IntentInformation> currentList = this.intentMap.get(classNameOrActionString);
			this.intentMap.remove(classNameOrActionString);
			if (currentList == null) {
				currentList = new ArrayList<>();
			}
			currentList.addAll(inIntents);
			this.intentMap.put(classNameOrActionString, currentList);
		}
	}

	public List<IntentFilter> getIntentFilters() {
		return this.appManifestInfo.getIntentFilters();
	}

	public void setIntentFilters(final List<IntentFilter> intentFilters) {
		this.appManifestInfo.setIntentFilters(intentFilters);
	}

	public void setAppManifestInfo(final ManifestInfo appManifestInfo) {
		this.appManifestInfo = appManifestInfo;
	}

	public ManifestInfo getAppManifestInfo() {
		return this.appManifestInfo;
	}

	@Override
	public String toString() {
		int i = 0;
		String str = i + ": " + getAppName() + "\n";
		i++;
		for (final SootClass sc : getAppClasses()) {
			str += i + ": " + "- " + sc.toString() + "\n";
			i++;
			for (final SootMethod sm : sc.getMethods()) {
				if (!sc.isInterface()) {
					if (sm.retrieveActiveBody() != null && sm.isDeclared()) {
						str += i + ": " + " - " + sm.toString() + "\n";
						i++;
						for (final Unit u : sm.retrieveActiveBody().getUnits()) {
							str += i + ": " + "  - " + u.toString() + "\n";
							i++;
						}
					}
				}
			}
		}
		return str;
	}
}
