package de.upb.pga3.panda2.core.services;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.upb.pga3.panda2.core.datastructures.JavaAPIMethod;
import de.upb.pga3.panda2.core.datastructures.Permission;
import de.upb.pga3.panda2.utilities.Constants;
import soot.util.HashMultiMap;
import soot.util.MultiMap;

/**
 * This Reader is created for reading results generated with the tool: PScout
 *
 * PScout: Analyzing the Android Permission Specification
 * http://pscout.csl.toronto.edu/
 *
 * @author Felix
 *
 */
public final class A3DataStorage implements DataStorage {
	// Instance replied in getInstance() method. (Singleton pattern)
	private static final A3DataStorage INSTANCE = new A3DataStorage();

	// Logger initialization
	private static final Logger LOGGER = LogManager.getLogger(A3DataStorage.class);

	// Data structure for mapping permissions
	private Map<HashSet<String>, List<Permission>> hashmap;
	private Map<String, Permission> allPermissions;

	// Result specification
	private Path folder;

	// Constants for possible PScout result outputs
	public final static String ALL_MAPPINGS = "allmappings";
	public final static String CONTENT_PROVIDER_FIELD_PERMISSION = "contentproviderfieldpermission";
	public final static String CONTENT_PROVIDER_PERMISSION = "contentproviderpermission";
	public final static String INTENT_PERMISSION = "intentpermission";
	public final static String PUBLISHED_API_MAPPING = "publishedapimapping";

	// Reader object used to read any file
	private BufferedReader bufferedReader;

	/*
	 * ====================================================================
	 * process for source and sink computer
	 */
	private MultiMap<String, JavaAPIMethod> mLstSources;
	private MultiMap<String, JavaAPIMethod> mLstSinks;

	// ====================================================================

	// List of Android call back classes
	private Set<String> androidCallBacks;

	/**
	 * Private constructor used for the singleton pattern
	 *
	 * @throws IOException
	 */
	private A3DataStorage() {
		// Initialize the data structures
		this.hashmap = new HashMap<>();
		this.allPermissions = new HashMap<>();

		// Set data path
		this.folder = Paths.get("data/");

		// Read all permissions
		try {
			this.bufferedReader = new BufferedReader(new FileReader(this.folder.toString() + "/listofallpermissions"));
			String curLine = "";
			this.bufferedReader.readLine(); // read first line which does not
											// contain a permission name
			while ((curLine = this.bufferedReader.readLine()) != null) {
				this.allPermissions.put(curLine, new Permission(curLine));
			}
			this.bufferedReader.close();
		} catch (final Exception e) {
			LOGGER.error("Error while reading " + this.folder.toString() + "/listofallpermissions" + " inputfile:\n"
					+ e.getMessage());
		}

		// Fill HashMap
		try {
			this.hashmap.putAll(readPScoutResult(ALL_MAPPINGS));
			this.hashmap.putAll(readPScoutResult(CONTENT_PROVIDER_FIELD_PERMISSION));
			this.hashmap.putAll(readPScoutResult(CONTENT_PROVIDER_PERMISSION));
			this.hashmap.putAll(readPScoutResult(INTENT_PERMISSION));
			this.hashmap.putAll(readPScoutResult(PUBLISHED_API_MAPPING));
		} catch (final IOException e) {
			LOGGER.error("An error accured while reading the input data:\n" + e.getMessage());
		}

		/*
		 * ====================================================================
		 * process for source and sink computer
		 */
		this.mLstSources = new HashMultiMap<>();
		this.mLstSinks = new HashMultiMap<>();
		this.androidCallBacks = new HashSet<>();
		try {
			this.mLstSources.putAll(readSourceSinkAPIMethods(Constants.SOURCE_FILE));
			this.mLstSinks.putAll(readSourceSinkAPIMethods(Constants.SINK_FILE));
			this.androidCallBacks = readCallBackClasses(Constants.ANDROID_CALL_BACK_FILE);
		} catch (final IOException e) {
			LOGGER.error("An error accured while reading the input data:\n" + e.getMessage());
		}
		// ====================================================================

		this.mLstSources.get("android.telephony.TelephonyManager");
		this.mLstSinks.get("android.telephony.SmsManager");

	}

	/**
	 * Returning always the same instance of an A3DataStorage object
	 *
	 * @return DataStorage object instance
	 */
	public static DataStorage getInstance() {
		return A3DataStorage.INSTANCE;
	}

	/**
	 * This method reads a part of the PScout result.
	 *
	 * @param type
	 *            Defines which part is read.
	 * @return Returns the map result for this part.
	 * @throws IOException
	 */
	private Map<HashSet<String>, List<Permission>> readPScoutResult(final String type) throws IOException {
		final HashMap<HashSet<String>, List<Permission>> hashmapToAdd = new HashMap<>();

		this.bufferedReader = new BufferedReader(new FileReader(this.folder.toString() + "/" + type));

		String curLine;
		Permission permission = null;
		HashSet<String> key;
		List<Permission> value;
		while ((curLine = this.bufferedReader.readLine()) != null) {
			if (type == A3DataStorage.ALL_MAPPINGS || type == A3DataStorage.PUBLISHED_API_MAPPING
					|| type == A3DataStorage.CONTENT_PROVIDER_FIELD_PERMISSION) {
				if (curLine.startsWith("<")) {
					key = new HashSet<>();
					key.add(curLine.substring(1, curLine.indexOf(": ")));
					key.add(curLine.substring(curLine.indexOf(" ", curLine.indexOf(": ") + 2) + 1,
							curLine.indexOf(">")));
					if (hashmapToAdd.get(key) == null) {
						value = new ArrayList<>();
						value.add(permission);
					} else {
						value = new ArrayList<>();
						value.addAll(hashmapToAdd.get(key));
						value.add(permission);
					}
					if (permission != null) {
						hashmapToAdd.put(key, value);
					} else {
						this.bufferedReader.close();
						throw new IOException("Found package, class and method before permission was defined in file: "
								+ this.folder.toString() + "/" + type);
					}
				} else if (curLine.startsWith("Permission:") || curLine.startsWith("PERMISSION:")) {
					permission = this.allPermissions.get(curLine.substring(11));
					if (permission == null) {
						permission = new Permission(curLine.substring(11));
						this.allPermissions.put(curLine.substring(11), permission);
					}
				}
			} else if (type == A3DataStorage.INTENT_PERMISSION) {
				key = new HashSet<>();
				key.add(curLine.substring(0, curLine.indexOf(" ")));
				permission = this.allPermissions.get(curLine.substring(curLine.indexOf(" ") + 1, curLine.length() - 2));
				if (permission == null) {
					permission = new Permission(curLine.substring(curLine.indexOf(" ") + 1, curLine.length() - 2));
					this.allPermissions.put(curLine.substring(curLine.indexOf(" ") + 1, curLine.length() - 2),
							permission);
				}
				value = new ArrayList<>();
				value.add(permission);
				hashmapToAdd.put(key, value);
			} else if (type == A3DataStorage.CONTENT_PROVIDER_PERMISSION) {
				curLine = curLine.replace(" pathPrefix", "");
				curLine = curLine.replace(" pathPattern", "");
				curLine = curLine.replace(" path", "");
				key = new HashSet<>();
				key.add(curLine.substring(0, curLine.indexOf(" ")));
				String permString = curLine.substring(curLine.indexOf(" ", curLine.indexOf(" ") + 1) + 1,
						curLine.length());
				permission = this.allPermissions.get(permString);
				if (permission == null) {
					if (permString.contains("grant-uri-permission")) {
						permString = "grant-uri-permission";
					}
					permission = new Permission(permString);
					this.allPermissions.put(permString, permission);
				}
				value = new ArrayList<>();
				value.add(permission);
				hashmapToAdd.put(key, value);
			}
		}

		this.bufferedReader.close();

		return hashmapToAdd;
	}

	@Override
	public List<Permission> mapAPICall(final String pkgClass, final String method) {
		final HashSet<String> needle = new HashSet<>();
		needle.add(pkgClass);
		needle.add(method);

		return this.hashmap.get(needle);
	}

	@Override
	public List<Permission> mapImplicitIntent(final String actionName) {
		final HashSet<String> needle = new HashSet<>();
		needle.add(actionName);

		return this.hashmap.get(needle);
	}

	@Override
	public List<Permission> mapContentProviderURI(final String uri) {
		final HashSet<String> needle = new HashSet<>();
		needle.add(uri);

		return this.hashmap.get(needle);
	}

	@Override
	public Map<String, Permission> getAllPermissions() {
		return this.allPermissions;
	}

	@Override
	public int getMaxAPILevel() {
		try {
			this.bufferedReader = new BufferedReader(new FileReader(this.folder.toString() + "/apilevel"));
			return Integer.valueOf(this.bufferedReader.readLine()).intValue();
		} catch (final NumberFormatException e) {
			LOGGER.error("Could not found API level information. " + this.folder.toString()
					+ "/apilevel seems to be corrupted.\n" + e.getMessage());
			return -1;
		} catch (final IOException e) {
			LOGGER.error(this.folder.toString() + " seems to be corrupted.\n" + e.getMessage());
			return -1;
		}
	}

	/*
	 * ====================================================================
	 * process for source and sink computer
	 */
	private MultiMap<String, JavaAPIMethod> readSourceSinkAPIMethods(final String inType) throws IOException {
		final MultiMap<String, JavaAPIMethod> mapMethods = new HashMultiMap<>();

		this.bufferedReader = new BufferedReader(new FileReader(this.folder.toString() + "/" + inType));
		String curLine = "";
		while ((curLine = this.bufferedReader.readLine()) != null) {
			// read each line in specified file
			final JavaAPIMethod method = parseMethod(curLine);
			if (method != null) {
				mapMethods.put(method.getClassName(), method);
			}
		}
		return mapMethods;
	}

	/**
	 * parse line of content for list of api methods
	 *
	 * @param inLineData
	 * @return
	 */
	private JavaAPIMethod parseMethod(final String inLineData) {
		if (inLineData != null && !inLineData.isEmpty()) {
			final String[] lstSubStr = inLineData.split("->");
			final int iLength = lstSubStr.length;
			if (iLength == 2) {
				String signature = lstSubStr[0].trim();
				final int pos = signature.lastIndexOf(">");
				signature = signature.substring(0, pos + 1);
				final JavaAPIMethod method = new JavaAPIMethod(signature);
				return method;
			}
		}
		return null;
	}

	/**
	 * check whether an API is source
	 *
	 * @param inClassName
	 * @param inSignature
	 * @return
	 */
	@Override
	public boolean isSource(final String inClassName, final String inSignature) {
		if (inSignature != null && !inSignature.isEmpty() && inClassName != null && !inClassName.isEmpty()) {
			// get list of source api or appropriate class
			final Set<JavaAPIMethod> setMethods = this.mLstSources.get(inClassName);

			if (setMethods != null && !setMethods.isEmpty()) {
				for (final JavaAPIMethod method : setMethods) {
					final String signature = method.getSignature();
					if (signature.equals(inSignature)) {
						return true;
					}
				}
			}
		}
		return false;
	}

	/**
	 * check whether an API is sink?
	 *
	 * @param inClassName
	 * @param inSignature
	 * @return
	 */
	@Override
	public boolean isSink(final String inClassName, final String inSignature) {

		if (inSignature != null && !inSignature.isEmpty() && inClassName != null && !inClassName.isEmpty()) {
			// get list of sink api or appropriate class
			final Set<JavaAPIMethod> setMethods = this.mLstSinks.get(inClassName);

			if (setMethods != null && !setMethods.isEmpty()) {
				for (final JavaAPIMethod method : setMethods) {
					final String signature = method.getSignature();
					if (signature.equals(inSignature)) {
						return true;
					}
				}
			}
		}

		return false;
	}

	// ====================================================================

	/**
	 * Method to parse android call back txt file
	 *
	 * @param fileType
	 *            - call back file name
	 * @return Set of android call back calsses
	 * @throws IOException
	 */

	private Set<String> readCallBackClasses(final String fileType) throws IOException {
		final Set<String> callBacks = new HashSet<>();
		this.bufferedReader = new BufferedReader(new FileReader(this.folder.toString() + "/" + fileType));
		String line;
		while ((line = this.bufferedReader.readLine()) != null) {
			if (!line.isEmpty()) {
				callBacks.add(line);
			}
		}
		this.bufferedReader.close();
		return callBacks;
	}

	/**
	 * Method returns call back classes
	 *
	 * @return Set of android call back classes.
	 */
	@Override
	public Set<String> getCallBackClasses() {
		return this.androidCallBacks;
	}
}
