package de.upb.pga3.panda2.core.services;

import java.util.ArrayList;
import java.util.List;

import soot.SootClass;
import soot.Unit;
import soot.ValueBox;

/**
 * this class is used to process for the global variables within classes
 *
 * @author nptsy
 */
public class GlobalVariable {
	private SootClass mParentClass;
	private String mVarName;
	private String mType;
	private List<String> mLstValues;
	private List<Unit> mLstUnits;

	/**
	 * get parent class
	 *
	 * @return a soot class which contains the global variable
	 */
	public SootClass getParentClass() {
		return this.mParentClass;
	}

	/**
	 * set the parent class for variables
	 *
	 * @param mParentClass
	 *            the class that contains the variable
	 */
	public void setParentClass(final SootClass mParentClass) {
		this.mParentClass = mParentClass;
	}

	/**
	 * get variable name
	 *
	 * @return string name of variable
	 */
	public String getVarName() {
		return this.mVarName;
	}

	/**
	 *
	 * @param mVarName
	 */
	public void setVarName(final String inVarName) {
		this.mVarName = inVarName;
	}

	/**
	 * get type of the varibale
	 *
	 * @return string type of variable
	 */
	public String getType() {
		return this.mType;
	}

	/**
	 * set type of variable
	 *
	 * @param mType
	 *            the type of variable
	 */
	public void setType(final String inType) {
		this.mType = inType;
	}

	/**
	 * add value to the list of values for the variable
	 *
	 * @param inValue
	 *            a new value of the variable
	 */
	private void addValue(final String inValue) {
		if (inValue != null && !inValue.isEmpty()) {
			this.mLstValues.add(inValue);
		}
	}

	/**
	 * get list of values
	 */
	public List<String> getValues() {
		return this.mLstValues;
	}

	/**
	 * add a new unit which is an assignment for the variable
	 *
	 * @param inUnit
	 *            the unit which assigns value to the variable
	 */
	public void addUnit(final Unit inUnit) {
		if (inUnit != null) {
			this.mLstUnits.add(inUnit);
			// processing for value
			final List<ValueBox> lstUsedBoxes = inUnit.getUseBoxes();
			if (lstUsedBoxes != null && lstUsedBoxes.size() >= 2) {
				addValue(lstUsedBoxes.get(1).getValue().toString());
			}

		}
	}

	/**
	 * Contrucstor
	 *
	 * @param parent
	 */
	public GlobalVariable(final SootClass parent) {
		this.mParentClass = parent;
		this.mVarName = "";
		this.mType = "";
		this.mLstValues = new ArrayList<>();
		this.mLstUnits = new ArrayList<>();
	}

	@Override
	public String toString() {
		return this.mParentClass.getName() + ": " + this.mType + " " + this.mVarName;
	}

	@Override
	public boolean equals(final Object inObj) {
		if (inObj != null && inObj instanceof GlobalVariable) {

			final String className = inObj.toString();
			if (!className.equals(this.mParentClass.toString())) {
				return false;
			}

			final GlobalVariable inVar = (GlobalVariable) inObj;

			if (this.mVarName != null && inVar.mVarName != null && !this.mVarName.equals(inVar.mVarName)) {
				return false;
			}

			if (inVar.mType != null && this.mType != null && !this.mType.equals(inVar.mType)) {
				return false;
			}
			return true;
		}
		return false;
	}

	/**
	 * check if the variable is in a specific type
	 *
	 * @param inString
	 *            the type that want to be checked
	 * @return true if the variable has the specified type, otherwise false
	 */
	public boolean isInTypeOf(final String inString) {
		if (inString != null && this.mType != null) {
			return this.mType.equals(inString);
		}
		return false;
	}

}
