package de.upb.pga3.panda2.extension;

import java.util.List;
import java.util.Map;

import de.upb.pga3.panda2.core.datastructures.EnhancedInput;
import de.upb.pga3.panda2.core.datastructures.Permission;
import de.upb.pga3.panda2.core.services.CoreServices;
import de.upb.pga3.panda2.core.services.DataStorage;
import de.upb.pga3.panda2.core.services.XMLParser;
import soot.SootClass;
import soot.Unit;
import soot.Value;
import soot.ValueBox;
import soot.jimple.internal.AbstractInvokeExpr;

/**
 * This class is used to map permissions to all kinds of elements.
 *
 * @author Felix
 */
public class PermissionMapper {
	EnhancedInput ei;

	// Information requested from the XMLParser
	private List<String> usesPermissions;
	private Map<String, String> requiredPermissions;

	// Information requested from the DataStorage
	private DataStorage dataStorage;
	private Map<String, Permission> allPermissions;

	/**
	 * Constructor that is getting all required information in order to map
	 * permissions to elements.
	 */
	PermissionMapper(final EnhancedInput ei) {
		this.ei = ei;

		this.dataStorage = CoreServices.getDataStorageInstance();
		this.allPermissions = this.dataStorage.getAllPermissions();

		final XMLParser parser = CoreServices.getXMLParserInstance();
		this.usesPermissions = parser.getUsesPermission();

		this.requiredPermissions = parser.getRequiredPermissions();
	}

	/**
	 * This method maps one or more permissions to an {@link App} element.
	 * Referring to the <uses-permission> tags in the Android manifest.
	 *
	 * @param element
	 *            Permissions will be added to this element.
	 */
	public void mapPermissionsToApp() {
		for (final String permissionStr : this.usesPermissions) {
			this.ei.addPermission(this.allPermissions.get(permissionStr));
		}
	}

	/**
	 * This method maps one or more permissions to an {@link Unit} element. By
	 * using the DataStorage.
	 *
	 * @param element
	 *            Permissions will be added to this element.
	 */
	public void mapPermissionsToStatement(final Unit element) {
		final String stmtStr = element.toString();
		if (stmtStr.contains("android.net.Uri parse(java.lang.String)")) {
			// Content provider
			String uriStr = null;
			for (final ValueBox item : element.getUseBoxes()) {
				final Value value = item.getValue();
				if (value instanceof AbstractInvokeExpr) {
					final AbstractInvokeExpr aExpr = (AbstractInvokeExpr) value;
					if (aExpr.getArgs() != null && !aExpr.getArgs().isEmpty()) {
						uriStr = ((AbstractInvokeExpr) value).getArg(0).toString();
						break;
					}
				}
			}
			if (uriStr != null) {
				uriStr = uriStr.substring(1, uriStr.length() - 1);
				final List<Permission> permList = this.dataStorage.mapContentProviderURI(uriStr);
				if (permList != null) {
					for (final Permission permission : permList) {
						this.ei.addPermissionTo(element, this.allPermissions.get(permission.getName()));
					}
				}
				this.ei.addToMaybeMoreList(element);
			}
		} else if (stmtStr.contains("<android.content.Intent:")) {
			// Implicit intent
			String actionStr = null;
			if (stmtStr.contains("void <init>(java.lang.String)")
					|| stmtStr.contains("android.content.Intent setAction(java.lang.String)")) {
				for (final ValueBox item : element.getUseBoxes()) {
					final Value value = item.getValue();
					if (value instanceof AbstractInvokeExpr) {
						final AbstractInvokeExpr aExpr = (AbstractInvokeExpr) value;
						if (aExpr.getArgs() != null && !aExpr.getArgs().isEmpty()) {
							actionStr = ((AbstractInvokeExpr) value).getArg(0).toString();
							break;
						}
					}
				}
				if (actionStr != null) {
					actionStr = actionStr.substring(1, actionStr.length() - 1);
					final List<Permission> permList = this.dataStorage.mapImplicitIntent(actionStr);
					if (permList != null) {
						for (final Permission permission : permList) {
							this.ei.addPermissionTo(element, this.allPermissions.get(permission.getName()));
						}
					}
					this.ei.addToMaybeMoreList(element);
				}
			}
		} else {
			// API call
			List<Permission> fetchedPermissionList = null;
			try {
				final String pkgName = stmtStr.substring(stmtStr.indexOf("<") + 1, stmtStr.indexOf(":"));
				final String method = stmtStr.substring(stmtStr.indexOf(" ", stmtStr.indexOf(":") + 3) + 1,
						stmtStr.indexOf(">"));
				fetchedPermissionList = this.dataStorage.mapAPICall(pkgName, method);
			} catch (final StringIndexOutOfBoundsException e) {
				return;
			}
			if (fetchedPermissionList == null || fetchedPermissionList.isEmpty()) {
				return;
			}
			for (final Permission permission : fetchedPermissionList) {
				this.ei.addPermissionTo(element, this.allPermissions.get(permission.getName()));
			}
		}
	}

	/**
	 * This method maps one or more permissions to an {@link SootClass} element.
	 * Referring to the android:permission entries in component definitions of
	 * the manifest.
	 *
	 * @param element
	 *            Permissions will be added to this element.
	 */
	public void mapPermissionsToComponent(final SootClass element) {
		if (this.allPermissions.get(this.requiredPermissions.get(element.toString())) != null) {
			this.ei.addPermissionTo(element, this.allPermissions.get(this.requiredPermissions.get(element.toString())));
		}
	}
}
