package de.upb.pga3.panda2.extension.lvl1.analyzer;

import java.util.Collection;

import de.upb.pga3.panda2.core.datastructures.AnalysisGraph;
import de.upb.pga3.panda2.core.datastructures.EnhancedInput;
import de.upb.pga3.panda2.core.datastructures.Message;
import de.upb.pga3.panda2.core.datastructures.MessageType;
import de.upb.pga3.panda2.core.datastructures.Permission;
import de.upb.pga3.panda2.core.datastructures.Transition;
import de.upb.pga3.panda2.core.services.CoreServices;
import de.upb.pga3.panda2.extension.Enhancer;
import de.upb.pga3.panda2.extension.lvl1.AnalysisResultLvl1;
import de.upb.pga3.panda2.extension.lvl1.ResultLeafLvl1;
import de.upb.pga3.panda2.extension.lvl1.ResultTreeLvl1;
import de.upb.pga3.panda2.extension.lvl1.ResultTypeLvl1;
import soot.Body;
import soot.SootClass;
import soot.SootMethod;
import soot.Unit;

/**
 * This class computes the main part of any Level 1 analysis. It will primarily
 * be creating an {@link AnalysisResultLvl1} object as analysis result.
 *
 * @author Felix
 *
 */
public class ManifestPermissionComparerLvl1 {
	private AnalysisGraph graph;
	private EnhancedInput ei;

	private Collection<Permission> allPermissions;

	/**
	 * In the following variables the result will be stored.
	 */
	private ResultLeafLvl1 resultApp;
	private ResultTreeLvl1 resultComponents;
	private ResultTreeLvl1 resultClasses;
	private ResultTreeLvl1 resultMethods;
	private AnalysisResultLvl1 anaResult;

	/**
	 * Flags for making sure that some messages are provided only once.
	 */
	boolean flagMissingMessage = false;
	boolean flagMaybeMissingMessage = false;
	boolean flagUnusedMessage = false;
	boolean flagMaybeRequiredMessage = false;

	/**
	 * Constructor initializing the analysis result parts.
	 */
	public ManifestPermissionComparerLvl1() {
		this.resultComponents = new ResultTreeLvl1();
		this.resultClasses = new ResultTreeLvl1();
		this.resultMethods = new ResultTreeLvl1();
	}

	/**
	 * This method will be executed in order to determine the Permission-Groups
	 * of all elements of the analyzed App.
	 *
	 * @param inGraph
	 *            All information needed will be provided by this graph.
	 * @return The finished analysis result will be replied.
	 */
	public AnalysisResultLvl1 compare(final AnalysisGraph inGraph) {
		this.graph = inGraph;
		this.ei = (EnhancedInput) this.graph.getInput();
		this.resultApp = new ResultLeafLvl1(this.ei.getAppName());
		this.allPermissions = CoreServices.getDataStorageInstance().getAllPermissions().values();
		fillUpPermissions();
		this.anaResult = new AnalysisResultLvl1(this.graph);
		this.anaResult.initialize(CoreServices.getXMLParserInstance());
		generateAnalysisResult();
		if (!this.ei.getMaybeMoreList().isEmpty()) {
			this.anaResult.addMessage(new Message(MessageType.SUGGESTION, "Suggestion: Deeper analysis",
					"There are some statements contained in the source code of this Application, that's permission use could not be determined.\nIn this case it is suggested to run a Level 2b analysis in order to get a more detailed result."));
		}

		return this.anaResult;
	}

	/**
	 * Fills the graph with permissions accordingly to the permissions assigned
	 * to statements by the {@link Enhancer}.
	 */
	private void fillUpPermissions() {
		for (final SootClass componentOrClass : this.ei.getAppClasses()) {
			for (final SootMethod method : componentOrClass.getMethods()) {
				if (!componentOrClass.isInterface()) {
					try {
						final Body body = method.retrieveActiveBody();
						for (final Unit stm : body.getUnits()) {
							if (this.ei.getMaybeMoreList().contains(stm)) {
								if (!this.ei.getMaybeMoreList().contains(this.ei)) {
									this.ei.addToMaybeMoreList(this.ei);
								}
								if (!this.ei.getMaybeMoreList().contains(method)) {
									this.ei.addToMaybeMoreList(method);
								}
								if (!this.ei.getMaybeMoreList().contains(componentOrClass)) {
									this.ei.addToMaybeMoreList(componentOrClass);
								}
							}
							if (this.ei.getPermissionsFor(stm) != null && !this.ei.getPermissionsFor(stm).isEmpty()) {
								for (final Permission stmtPerm : this.ei.getPermissionsFor(stm)) {
									if (!this.ei.getPermissionsFor(method).contains(stmtPerm)) {
										this.ei.addPermissionTo(method, stmtPerm);
									}
									if (!this.ei.getPermissionsFor(componentOrClass).contains(stmtPerm)) {
										this.ei.addPermissionTo(componentOrClass, stmtPerm);
									}
								}
							}
						}
					} catch (final RuntimeException e) {
						if (e.getMessage().contains("No method source set for method")) {
							// Message was already given by the Enhancer. Do
							// nothing.
						} else {
							e.printStackTrace();
						}
					}
				}
			}
		}
	}

	/**
	 * This method sorts the permissions into the 5 Permission-Groups
	 */
	private void generateAnalysisResult() {
		// Iterate over classes (components)
		for (final SootClass classOrComponentElement : this.ei.getAppClasses()) {
			// Iterate over methods
			if (!classOrComponentElement.isInterface()) {
				for (final SootMethod methodElement : classOrComponentElement.getMethods()) {
					checkElement(methodElement);
				}
				checkElement(classOrComponentElement);
			}
		}
		checkElement(this.ei);

		// Set partial results in result object
		this.anaResult.setApp(this.resultApp);
		this.anaResult.setComponents(this.resultComponents);
		this.anaResult.setClasses(this.resultClasses);
		this.anaResult.setMethods(this.resultMethods);
	}

	/**
	 * Assigns a Permission-Group to one specific element.
	 *
	 * @param element
	 *            The Permissin-Group will be assigned to this element.
	 */
	private void checkElement(final Object element) {
		// flags
		boolean hasPermission;
		boolean assignedToDescendant;
		boolean inMaybeMore = false;

		for (final Permission permission : this.allPermissions) {
			// Check if app has current permission assigned
			if (this.ei.getPermissions().contains(permission)) {
				hasPermission = true;
			} else {
				hasPermission = false;
			}

			// Check if any child has assigned that permission..
			assignedToDescendant = false;
			if (this.ei.getChildren(element) != null) {
				for (final Object childElement : this.ei.getChildren(element)) {
					if (assignedToChild(childElement, permission)) {
						assignedToDescendant = true;
						break;
					}
				}
			}

			// If needed, check if element is in maybeMore list
			if (!(hasPermission && assignedToDescendant || !hasPermission && assignedToDescendant)) {
				if (this.ei.getMaybeMoreList().contains(element)) {
					inMaybeMore = true;
				}
			}

			// Set result accordingly to the set of flags
			setInResult(element, permission, hasPermission, assignedToDescendant, inMaybeMore);
		}
	}

	/**
	 * Checks whether the permission is assigned to any child of this child
	 * element.
	 *
	 * @param childElement
	 *            The element to investigate further.
	 * @param permission
	 *            The {@link Permission} to check
	 * @return If the {@link Permission} is assigned to any child true will be
	 *         returned. In any other case false.
	 */
	private boolean assignedToChild(final Object childElement, final Permission permission) {
		for (final Permission childPermission : this.ei.getPermissions(childElement)) {
			if (childPermission == permission) {
				return true;
			}
		}
		// ..or contains an explicit intent to a target that has been assigned
		// that permission.
		if (childElement instanceof Unit) {
			if (this.graph.getOutgoingTransitions(childElement) != null) {
				for (final Transition t : this.graph.getOutgoingTransitions(childElement)) {
					for (final Permission permissionFromTarget : this.ei.getPermissionsFor((SootClass) t.getTarget())) {
						if (permissionFromTarget == permission) {
							return true;
						}
					}
				}
			}
		} else {
			// e.g. in case of an interface
			if (this.ei.getChildren(childElement) != null) {
				for (final Object childsChildElement : this.ei.getChildren(childElement)) {
					if (assignedToChild(childsChildElement, permission)) {
						return true;
					}
				}
			}
		}
		return false;
	}

	/**
	 * Determines the Permission-Group.
	 *
	 * @param element
	 *            The Permission-Group will be determined for this element
	 * @param permission
	 *            The Permission-Group will be assigned to this
	 *            {@link Permission}.
	 * @param hasPermission
	 *            Flag representing the result of the check if the
	 *            {@link Permission} was assigned to this element.
	 * @param assignedToChild
	 *            Flag representing the result of the check if the
	 *            {@link Permission} was assigned to any child of this element.
	 * @param inMaybeMore
	 *            Flag representing the result of the check if this elements
	 *            belongs to the maybemore list or not.
	 */
	private void setInResult(final Object element, final Permission permission, final boolean hasPermission,
			final boolean assignedToChild, final boolean inMaybeMore) {
		if (hasPermission) {
			if (assignedToChild) {
				setInResult(element, permission, ResultTypeLvl1.REQUIRED);
				return;
			} else {
				if (inMaybeMore) {
					setInResult(element, permission, ResultTypeLvl1.MAYBE_REQUIRED);
					if (!this.flagMaybeRequiredMessage) {
						this.anaResult.addMessage(new Message(MessageType.WARNING, "MAYBE REQUIRED Permission",
								"The analyzed App contains one or more maybe required permissions. There might be a cooperation between this App and another that leads to a intended permission usage."));
						this.flagMaybeRequiredMessage = true;
					}
					return;
				} else {
					setInResult(element, permission, ResultTypeLvl1.UNUSED);
					if (!this.flagUnusedMessage) {
						this.anaResult.addMessage(new Message(MessageType.WARNING, "UNUSED Permission",
								"The analyzed App contains one or more unused permissions. Permissions are declared in the Android manifest but never used."));
						this.flagUnusedMessage = true;
					}
					return;
				}
			}
		} else {
			if (assignedToChild) {
				setInResult(element, permission, ResultTypeLvl1.MISSING);
				if (!this.flagMissingMessage) {
					this.anaResult.addMessage(new Message(MessageType.ERROR, "MISSING Permission",
							"The analyzed App contains one or more missing permissions. This should be considered as security critical and might lead to a data leak."));
					this.flagMissingMessage = true;
				}
				return;
			} else {
				if (inMaybeMore) {
					setInResult(element, permission, ResultTypeLvl1.MAYBE_MISSING);
					if (!this.flagMaybeMissingMessage) {
						this.anaResult.addMessage(new Message(MessageType.WARNING, "MAYBE MISSING Permission",
								"The analyzed App contains one or more maybe missing permissions. There might be a cooperation between this App and another that leads to a unintended permission usage."));
						this.flagMaybeMissingMessage = true;
					}
					return;
				}
			}
		}
	}

	/**
	 * Will set the final Permission-Group for this pair of element and
	 * {@link Permission}.
	 *
	 * @param element
	 *            The Permission-Group will be determined for this element
	 * @param permission
	 *            The Permission-Group will be assigned to this
	 *            {@link Permission}.
	 * @param group
	 *            The determined Permission-Group.
	 */
	private void setInResult(final Object element, final Permission permission, final int group) {
		if (element instanceof EnhancedInput) {
			this.resultApp.addPermission(permission, group);
		} else if (element instanceof SootClass) {
			if (this.ei.isAndroidComponent((SootClass) element)) {
				ResultLeafLvl1 tempLeaf = this.resultComponents.getLeaf(element.toString());
				if (tempLeaf == null) {
					tempLeaf = new ResultLeafLvl1(element.toString());
					this.resultComponents.addLeaf(tempLeaf);
				}
				tempLeaf.addPermission(permission, group);
			}
			ResultLeafLvl1 tempLeaf = this.resultClasses.getLeaf(element.toString());
			if (tempLeaf == null) {
				tempLeaf = new ResultLeafLvl1(element.toString());
				this.resultClasses.addLeaf(tempLeaf);
			}
			tempLeaf.addPermission(permission, group);
		} else {
			ResultLeafLvl1 tempLeaf = this.resultMethods.getLeaf(element.toString());
			if (tempLeaf == null) {
				tempLeaf = new ResultLeafLvl1(element.toString());
				this.resultMethods.addLeaf(tempLeaf);
			}
			tempLeaf.addPermission(permission, group);
		}
	}
}
