/**
 *
 */
package de.upb.pga3.panda2.extension.lvl2a;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.upb.pga3.panda2.core.datastructures.AnalysisResult;
import de.upb.pga3.panda2.core.datastructures.DetailLevel;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.ClassElement;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.FlowPath;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.MethodElement;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.PathElement;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.ResourceElement;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.StatementElement;
import de.upb.pga3.panda2.utilities.Constants;
import de.upb.pga3.panda2.utilities.HTMLFrameBuilder;

/**
 * AnalysisResult for level 2a
 *
 * @author nptsy
 * @author Fabian
 */
public class AnalysisResultLvl2a extends AnalysisResult {
	static final Logger LOGGER = LogManager.getLogger(AnalysisResultLvl2a.class);

	/**
	 * DetailLevelLvl2a enumeration
	 *
	 * @author Fabian
	 */
	public enum DetailLevelLvl2a implements DetailLevel {
		RES_TO_RES, COMPONENT, METHOD, STATEMENT;
	}

	// map of available paths from source
	// MultiMap<Permission, FlowPath> mMapSources;
	Map<ResourceElement, List<FlowPath>> mMapSources;
	// map of available paths to sink
	// MultiMap<Permission, FlowPath> mMapSinks;
	Map<ResourceElement, List<FlowPath>> mMapSinks;

	// all filters for the user
	Map<String, ResourceElement> mMapFilters;

	/**
	 * Construtor
	 */
	public AnalysisResultLvl2a() {

		// initialize list and map here
		this.mMapFilters = new TreeMap<>();
		// this.mMapSinks = new HashMultiMap<>();
		// this.mMapSources = new HashMultiMap<>();
		this.mMapSinks = new HashMap<>();
		this.mMapSources = new HashMap<>();
	}

	/**
	 * add a flow path to the map of result
	 *
	 * @param inPath
	 *            the new flow path
	 * @return true if the new flow path is added to the map result, otherwise
	 *         false
	 */
	public boolean addPath(final FlowPath inPath) {

		if (inPath != null && inPath.isComplete()) {
			// add to map sources
			List<FlowPath> lstFlowPaths = this.mMapSources.get(inPath.getSource());
			if (lstFlowPaths == null) {
				lstFlowPaths = new ArrayList<>();
				lstFlowPaths.add(inPath);
				this.mMapSources.put(inPath.getSource(), lstFlowPaths);
			} else {
				lstFlowPaths.add(inPath);
			}

			// add to map sinks
			List<FlowPath> lstFlowPaths2 = this.mMapSinks.get(inPath.getSink());
			if (lstFlowPaths2 == null) {
				lstFlowPaths2 = new ArrayList<>();
				lstFlowPaths2.add(inPath);
				this.mMapSinks.put(inPath.getSink(), lstFlowPaths2);
			} else {
				lstFlowPaths2.add(inPath);
			}

			return true;
		} else {
			throw new IllegalArgumentException("The path to be added must not be null and must be complete!");
		}

	}

	/**
	 * get a list of flow paths for a specific source permission
	 *
	 * @param srcPerm
	 *            the source permission
	 * @return a list of flow paths
	 */
	public List<FlowPath> getPathsForSource(final ResourceElement srcPerm) {
		return this.mMapSources.get(srcPerm);
	}

	/**
	 * get a list of flow paths for a specific sink permission
	 *
	 * @param sinkPerm
	 *            the sink permission
	 * @return a list of flow paths
	 */
	public List<FlowPath> getPathsForSink(final ResourceElement sinkPerm) {
		return this.mMapSinks.get(sinkPerm);
	}

	/**
	 * get a set of flow paths existing as results
	 *
	 * @return
	 */
	public Set<FlowPath> getPaths() {
		final Collection<List<FlowPath>> paths = this.mMapSources.values();
		final Set<FlowPath> setFlowPaths = new HashSet<>();
		for (final List<FlowPath> lstFlowPaths : paths) {
			if (lstFlowPaths != null) {
				setFlowPaths.addAll(lstFlowPaths);
			}
		}
		return setFlowPaths;
	}

	/**
	 * Creates a set of filter strings for the user based on the paths added
	 */
	public void createFilters() {

		for (final ResourceElement p : this.mMapSources.keySet()) {
			this.mMapFilters.put(Constants.PREFIX_SOURCE + p.toString(), p);
		}
		for (final ResourceElement p : this.mMapSinks.keySet()) {
			this.mMapFilters.put(Constants.PREFIX_SINK + p.toString(), p);
		}
	}

	/**
	 * add sources to the existing map of results
	 *
	 * @param inSources
	 *            the new sources
	 */
	public void addSources(final Collection<ResourceElement> inSources) {

		if (inSources != null) {
			for (final ResourceElement p : inSources) {
				if (!this.mMapSources.containsKey(p)) {
					this.mMapSources.put(p, null);
				}
			}
		}
	}

	/**
	 * add a sinks to the existing map of results
	 *
	 * @param inSinks
	 *            the new sinks
	 */
	public void addSinks(final Collection<ResourceElement> inSinks) {

		if (inSinks != null) {
			for (final ResourceElement p : inSinks) {
				if (!this.mMapSinks.containsKey(p)) {
					this.mMapSinks.put(p, null);
				}
			}
		}
	}

	/**
	 * create textual result of analysis
	 *
	 * @param inPaths
	 *            list of paths
	 * @return a string value in textual result
	 */
	private String createTextualResult(final Collection<FlowPath> inPaths, final DetailLevel inDetailLvl,
			final boolean autoHideHeader) {

		final StringBuilder sb = new StringBuilder();
		final HTMLFrameBuilder fb = new HTMLFrameBuilder(sb, "Intra-App Information Flow Analysis");
		/*
		 * in case no flow path between sources and sinks exist ==> deciding
		 * trustworthy or not here
		 */
		fb.setAppTrustworthy(inPaths.size() == 0);

		fb.addStaticticsRow("Number of source permission(s): ",
				"<font color=\"red\"><strong>" + this.mMapSources.size() + "</strong></font>");
		fb.addStaticticsRow("Number of sink permission(s): ",
				"<font color=\"blue\"><strong>" + this.mMapSinks.size() + "</strong></font>");
		fb.addStaticticsRow("Number of distinguish path(s): ", "<strong>" + inPaths.size() + "</strong>");
		fb.addLegendEntry("<div class=\"icon\" style=\"background:#ff0000;\"></div>", "SOURCE PERMISSION");
		fb.addLegendEntry("<div class=\"icon\" style=\"background:#0006fd;\"></div>", "SINK PERMISSION");

		fb.setHeaderAutoHide(autoHideHeader);
		fb.setHintPersistent(false);
		fb.setCustomStyle(HtmlTableBuilder.cssStyle);

		final HtmlTableBuilder tableBuilder = new HtmlTableBuilder(false, getAppName());

		tableBuilder.processResult(inPaths, inDetailLvl);

		try {
			fb.append(tableBuilder.toString());
			fb.complete();
		} catch (final IOException e) {
			e.printStackTrace();
		}

		return sb.toString();
	}

	/**
	 * create graphical result of analysis
	 *
	 * @param inPaths
	 *            list of paths
	 * @param inDetailLvl
	 * @return a string value of graphical result
	 */
	private String createGraphicalResult(final Collection<FlowPath> inPaths, final DetailLevelLvl2a inDetailLvl,
			final boolean autoHideHeader) {

		final StringBuilder sb = new StringBuilder();
		final HTMLFrameBuilder fb = new HTMLFrameBuilder(sb, "Intra-App Information Flow Analysis");

		fb.setAppTrustworthy(this.mMapSources.size() == 0);
		fb.addStaticticsRow("Number of source permission(s): ", "<strong>" + this.mMapSources.size() + "</strong>");
		fb.addStaticticsRow("Number of sink permission(s): ", "<strong>" + this.mMapSinks.size() + "</strong>");
		fb.addStaticticsRow("Number of distinguish path(s): ", "<strong>" + inPaths.size() + "</strong>");

		createLegend(inDetailLvl, fb);
		fb.addLegendEntry("<div style=\"background:" + SVGGraphBuilder.DEF_PATH_COLOR + ";\"></div>",
				"Information Flow Edge");

		fb.setHeaderAutoHide(autoHideHeader);
		fb.setHintPersistent(false);
		fb.setCustomStyle(SVGGraphBuilder.GRAPH_STYLE);
		fb.setCustomScript(SVGGraphBuilder.GRAPH_SCRIPT);

		final SVGGraphBuilder builder = new SVGGraphBuilder();

		for (final FlowPath path : inPaths) {
			builder.addPath(path, SVGGraphBuilder.DEF_PATH_COLOR);
		}

		try {
			fb.append(SVGGraphBuilder.SVG_FILTER);
			fb.append(builder.buildGraph());
			fb.complete();
		} catch (final IOException e) {
			LOGGER.error("Error when creating graphical result: {}", e.getMessage());
			LOGGER.debug(e);
		}

		return sb.toString();
	}

	static void createLegend(final DetailLevelLvl2a inDetailLvl, final HTMLFrameBuilder fb) {

		fb.addLegendEntry("<div style=\"background:" + SVGGraphBuilder.RESOURCE_COLOR + ";\"></div>", "Resource");
		if (inDetailLvl.compareTo(DetailLevelLvl2a.COMPONENT) >= 0) {
			fb.addLegendEntry("<div style=\"background:" + SVGGraphBuilder.CLASS_COLOR + ";\"></div>", "Class");
		}
		if (inDetailLvl.compareTo(DetailLevelLvl2a.METHOD) >= 0) {
			fb.addLegendEntry("<div style=\"background:" + SVGGraphBuilder.METHOD_COLOR + ";\"></div>", "Method");
		}
		if (inDetailLvl.compareTo(DetailLevelLvl2a.STATEMENT) >= 0) {
			fb.addLegendEntry("<div style=\"background:" + SVGGraphBuilder.STATEMENT_COLOR + ";\"></div>", "Statement");
		}
		if (inDetailLvl.equals(DetailLevelLvl2a.METHOD)) {
			fb.addLegendEntry(SVGGraphBuilder.LEGEND_ICON_CIRCLE, "Class node");
		} else if (inDetailLvl.equals(DetailLevelLvl2a.STATEMENT)) {
			fb.addLegendEntry(SVGGraphBuilder.LEGEND_ICON_CIRCLE, "Class/method node");
		}
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see
	 * de.upb.pga3.panda2.core.datastructures.AnalysisResult#getGraphicalResult
	 * (de.upb.pga3.panda2.core.datastructures.DetailLevel, java.util.List)
	 */
	@Override
	public String getGraphicalResult(final DetailLevel inDetailLvl, final List<String> inFilters,
			final boolean inShowStats) {

		final Collection<FlowPath> pathsForRep = preprocess(inDetailLvl, inFilters);
		return createGraphicalResult(pathsForRep, (DetailLevelLvl2a) inDetailLvl, inShowStats);
	}

	private Collection<FlowPath> preprocess(final DetailLevel inDetailLvl, final List<String> inFilters) {

		// validate the detail level
		// preprocess for creating graphical result
		RepresentationPreprocessor rp;
		if (inDetailLvl == null || !(inDetailLvl instanceof DetailLevelLvl2a)) {
			LOGGER.warn("No detail level is specified. Use default!");
			rp = new RepresentationPreprocessor(this.mMapSources, this.mMapSinks, this.mMapFilters, inFilters,
					DetailLevelLvl2a.RES_TO_RES);
		} else {
			rp = new RepresentationPreprocessor(this.mMapSources, this.mMapSinks, this.mMapFilters, inFilters,
					(DetailLevelLvl2a) inDetailLvl);
		}
		return rp.preprocess();
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see
	 * de.upb.pga3.panda2.core.datastructures.AnalysisResult#getTextualResult
	 * (de.upb.pga3.panda2.core.datastructures.DetailLevel, java.util.List)
	 */
	@Override
	public String getTextualResult(final DetailLevel inDetailLvl, final List<String> inFilters,
			final boolean inShowStats) {

		final Collection<FlowPath> pathsForRep = preprocess(inDetailLvl, inFilters);
		return createTextualResult(pathsForRep, inDetailLvl, inShowStats);
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see
	 * de.upb.pga3.panda2.core.datastructures.AnalysisResult#getDetailLevels()
	 */
	@Override
	public List<DetailLevel> getDetailLevels() {

		final List<DetailLevel> levels = new ArrayList<>(DetailLevelLvl2a.values().length);
		levels.addAll(Arrays.asList(DetailLevelLvl2a.values()));
		return levels;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see de.upb.pga3.panda2.core.datastructures.AnalysisResult#getFilters()
	 */
	@Override
	public List<String> getFilters() {

		final List<String> filters = new ArrayList<>();
		filters.addAll(this.mMapFilters.keySet());
		return filters;
	}

	static class RepresentationPreprocessor {

		private final List<String> mFilters;
		private final DetailLevelLvl2a mDetailLvl;
		private final Map<ResourceElement, List<FlowPath>> mMapSources;
		private final Map<ResourceElement, List<FlowPath>> mMapSinks;
		private final Map<String, ResourceElement> mMapFilters;

		RepresentationPreprocessor(final Map<ResourceElement, List<FlowPath>> mapSources,
				final Map<ResourceElement, List<FlowPath>> mapSinks, final Map<String, ResourceElement> mapFilters,
				final List<String> filters, final DetailLevelLvl2a detailLvl) {
			this.mMapSources = mapSources;
			this.mMapSinks = mapSinks;
			this.mMapFilters = mapFilters;
			this.mFilters = filters;
			this.mDetailLvl = detailLvl;
		}

		Collection<FlowPath> preprocess() {

			final Collection<FlowPath> filteredPaths = filterPaths();
			LOGGER.debug("Selected {} paths after applying filters", filteredPaths.size());
			final Collection<FlowPath> preprocessedPaths = new LinkedHashSet<>(filteredPaths.size());
			for (final FlowPath p : filteredPaths) {
				preprocessedPaths.add(fitToDetailLvl(p));
			}
			LOGGER.debug("Selected {} paths after applying detail level {}", preprocessedPaths.size(),
					this.mDetailLvl.name());
			return preprocessedPaths;
		}

		/**
		 * Collects all {@link FlowPath}s in this result according to the
		 * provided filter strings. If no filters are provided, all paths will
		 * be returned. If a filter string is invalid, it will be skipped.
		 *
		 * @param inFilters
		 *            Filter strings to get paths for
		 * @return Collection of paths according to filters
		 */
		private Collection<FlowPath> filterPaths() {

			// The order of paths in the returned list is not deterministic
			// because of the use of HashMap

			// validate the list of filters
			if (this.mFilters == null || this.mFilters.isEmpty()
					|| this.mMapFilters.keySet().equals(new HashSet<>(this.mFilters))) {
				// return all paths
				final Set<FlowPath> setFlowPaths = new LinkedHashSet<>();
				final Collection<List<FlowPath>> colLstFlowPaths = this.mMapSinks.values();
				for (final List<FlowPath> lstFlowPaths : colLstFlowPaths) {
					if (lstFlowPaths != null) {
						setFlowPaths.addAll(lstFlowPaths);
					}
				}
				return setFlowPaths;
			}

			// Collect paths according to filters
			ResourceElement perm = null;
			final Set<FlowPath> setFlowPaths = new LinkedHashSet<>();
			for (final String filPerm : this.mFilters) {
				perm = this.mMapFilters.get(filPerm);
				if (perm != null) {
					if (filPerm.startsWith(Constants.PREFIX_SINK)) {
						setFlowPaths.addAll(this.mMapSinks.get(perm));
					} else if (filPerm.startsWith(Constants.PREFIX_SOURCE)) {
						setFlowPaths.addAll(this.mMapSources.get(perm));
					} else {
						LOGGER.warn("Filter string '{}' is invalid! Skipping...", filPerm);
					}
				} else {
					LOGGER.warn("Filter string '{}' is invalid! Skipping...", filPerm);
				}
			}
			return new ArrayList<>(setFlowPaths);
		}

		private FlowPath fitToDetailLvl(final FlowPath path) {

			if (this.mDetailLvl == null || this.mDetailLvl.equals(DetailLevelLvl2a.STATEMENT)) {
				return path;
			}

			if (this.mDetailLvl.equals(DetailLevelLvl2a.RES_TO_RES)) {
				final FlowPath newPath = new FlowPath();
				newPath.addElement(path.getSource());
				newPath.addElement(path.getSink());
				return newPath;
			}

			// mDetailLvl = COMPONENT or mDetailLvl = METHOD
			final FlowPath newPath = new FlowPath();
			PathElement ele1 = convertToLvl(path.getElement(0));
			newPath.addElement(ele1);
			PathElement ele2;
			for (int i = 1; i < path.getLength(); i++) {
				ele2 = convertToLvl(path.getElement(i));
				if (ele2 != null && !ele2.equals(ele1)) {
					newPath.addElement(ele2);
					ele1 = ele2;
				}
			}
			return newPath;
		}

		private PathElement convertToLvl(final PathElement ele) {

			if (this.mDetailLvl.equals(DetailLevelLvl2a.COMPONENT)) {
				return convertToLvlComponent(ele);
			} else if (this.mDetailLvl.equals(DetailLevelLvl2a.METHOD)) {
				return convertToLvlMethod(ele);
			} else {
				return ele;
			}
		}

		private static PathElement convertToLvlMethod(final PathElement ele) {

			if (ele instanceof ResourceElement || ele instanceof ClassElement || ele instanceof MethodElement) {
				return ele;
			} else if (ele instanceof StatementElement) {
				return ((StatementElement) ele).getMethodElement();
			} else {
				throw new IllegalArgumentException("Did not expect vertex of type " + ele.getClass());
			}
		}

		private static PathElement convertToLvlComponent(final PathElement ele) {

			if (ele instanceof ResourceElement) {
				return ele;
			} else if (ele instanceof ClassElement) {
				if (((ClassElement) ele).isAndroidComponent()) {
					return ele;
				} else {
					return null;
				}
			} else if (ele instanceof MethodElement) {
				return convertToLvlComponent(((MethodElement) ele).getClassElement());
			} else if (ele instanceof StatementElement) {
				return convertToLvlComponent(((StatementElement) ele).getMethodElement());
			} else {
				throw new IllegalArgumentException("Did not expect vertex of type " + ele.getClass());
			}
		}
	}
}
