package de.upb.pga3.panda2.extension.lvl2a;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.mxgraph.canvas.mxSvgCanvas;
import com.mxgraph.layout.mxGraphLayout;
import com.mxgraph.layout.mxOrganicLayout;
import com.mxgraph.model.mxCell;
import com.mxgraph.util.mxConstants;
import com.mxgraph.util.mxRectangle;
import com.mxgraph.util.mxXmlUtils;
import com.mxgraph.view.mxGraph;
import com.mxgraph.view.mxStylesheet;

import de.upb.pga3.panda2.extension.lvl2a.flowpath.ClassElement;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.FlowPath;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.MethodElement;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.PathElement;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.ResourceElement;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.StatementElement;
import soot.toolkits.scalar.Pair;

/**
 *
 * @author Fabian
 *
 */
public class SVGGraphBuilder {

	private static final Logger LOGGER = LogManager.getLogger(SVGGraphBuilder.class);

	private static final String GRAPH_STYLE_NAME = "GRAPH_STYLE";

	public static final String CLASS_COLOR = "#2691DE";// #3498DB
	public static final String METHOD_COLOR = "#1E77BF";// #2980B9
	public static final String STATEMENT_COLOR = "#0766B7";// #096EB0
	public static final String RESOURCE_COLOR = "#EDC010";// #F1C40F

	public static final String DEF_PATH_COLOR = "#203A57";// #2C3E50
	public static final String NEW_PATH_COLOR = "#21C87B";// #2ECC71
	public static final String REMOVED_PATH_COLOR = "#E04741";// #E74C3C
	public static final String HIGHLIGHTED_PATH_COLOR = "#F09514";// #F39C12

	private static final double SVG_GRAPH_EXTENSION = 50;

	private static final String PATH_HIGHLIGHTED_CLASS = "highlighted";

	public static final String GRAPH_STYLE = "svg {" + "font-size: 80%;" + "} " +

			"svg g.edge > path {" + "stroke-width: .3em;" + "transition: .2s ease-in-out;" + "} " +

			"svg g.cluster > text, " + "svg g.node > text {" + "fill: white;" + "stroke: black;"
			+ "stroke-width: .015em;" + "font-weight: bold;" + "} " +

			"svg g.edge > text {" + "transition: .2s ease-in-out;" + "} " +

			"svg g.edge." + PATH_HIGHLIGHTED_CLASS + " > polygon, " + "svg g.edge." + PATH_HIGHLIGHTED_CLASS
			+ " > path {" + "stroke-width: .4em;" + "stroke: " + HIGHLIGHTED_PATH_COLOR + ";"
			+ "filter: url(#drop-shadow);" + "} " +

			"svg g.edge." + PATH_HIGHLIGHTED_CLASS + " > text {" + "fill: " + HIGHLIGHTED_PATH_COLOR + ";"
			+ "filter: url(#drop-shadow);" + "} ";

	public static final String GRAPH_SCRIPT = "var edges;" +

			"function stylePath(edge) {" + "var edgeText = edge.parentNode.getElementsByTagName('text')[0];"
			+ "var pathIdx = edgeText.innerHTML;" + "var newPathIdx;" + "for (i = 0; i < edges.length; i++) {"
			+ "edgeText = edges[i].parentNode.getElementsByTagName('text')[0];" + "newPathIdx = edgeText.innerHTML;"
			+ "if(newPathIdx===pathIdx) {" + "edges[i].parentNode.classList.toggle('" + PATH_HIGHLIGHTED_CLASS + "');"
			+ "} " + "} " + "} " +

			"function edgeClicked(event) {" + "stylePath(event.target);" + "} " +

			"function setMousePathListener() {"
			+ "edges = document.querySelectorAll('svg g.edge > path');console.log('Found edges: '+edges.length);"
			+ "for (i = 0; i < edges.length; i++) {" + "edges[i].addEventListener('click', edgeClicked, false);" + "} "
			+ "} " +

			"window.addEventListener('DOMContentLoaded', setMousePathListener, false);";

	public static final String SVG_FILTER = "<svg height=\"0\" width=\"0\" xmlns=\"http://www.w3.org/2000/svg\">"
			+ "<filter id=\"drop-shadow\">" + "<feGaussianBlur in=\"SourceAlpha\" stdDeviation=\"2.2\"/>"
			+ "<feOffset dx=\"0\" dy=\"0\" result=\"offsetblur\"/>" + "<feFlood flood-color=\"white\"/>"
			+ "<feComposite in2=\"offsetblur\" operator=\"in\"/>" + "<feMerge>" + "<feMergeNode/>"
			+ "<feMergeNode in=\"SourceGraphic\"/>" + "</feMerge>" + "</filter>" + "</svg>";

	public static final String LEGEND_ICON_CIRCLE = "<svg viewBox=\"0 0 100 100\"><circle style=\"fill:#000000;stroke:none\" cx=\"50\" cy=\"50\" r=\"50\" /></svg>";

	private final Map<PathElement, Object> mVertexMap;

	private final Set<Pair<FlowPath, String>> mPaths;

	private int mPathIndex;

	public SVGGraphBuilder() {

		this.mVertexMap = new HashMap<>();
		this.mPathIndex = 1;
		this.mPaths = new HashSet<>();
	}

	public void addPath(final FlowPath p, final String color) {

		if (p == null) {
			throw new IllegalArgumentException("FlowPath p must not be null!");
		}
		if (color == null) {
			throw new IllegalArgumentException("color must not be null!");
		}

		final boolean added = this.mPaths.add(new Pair<>(p, color));
		if (!added) {
			LOGGER.trace("Path already extists in SVGGraphBuilder: {}", p.toString());
		}
	}

	private void addPathToGraph(final mxGraph graph, final Pair<FlowPath, String> path) {

		final FlowPath p = path.getO1();
		if (p.getLength() == 0) {
			throw new IllegalArgumentException("Path must not be empty!");
		}
		Object v1 = getVertex(graph, p.getElement(0));
		Object v2;
		for (int i = 1; i < p.getLength(); i++) {
			v2 = getVertex(graph, p.getElement(i));
			insertEdge(graph, v1, v2, path.getO2(), this.mPathIndex);
			v1 = v2;
		}
		this.mPathIndex++;
	}

	public String buildGraph() {

		final mxGraph graph = initGraph();

		graph.getModel().beginUpdate();
		for (final Pair<FlowPath, String> ePath : this.mPaths) {
			addPathToGraph(graph, ePath);
		}
		graph.getModel().endUpdate();

		LOGGER.debug("Building SVG graph containing {} paths", this.mPathIndex - 1);

		String svgGraph = null;
		try {
			svgGraph = buildGraphWithGraphviz(graph);
		} catch (final Exception e) {
			LOGGER.debug("Could not use Graphviz for rendering ({}), using JGraphX as fallback", e.getMessage());
			svgGraph = buildGraphWithJGraphX(graph);
		}
		return svgGraph;
	}

	private static String buildGraphWithGraphviz(final mxGraph graph) throws IOException, InterruptedException {

		final Runtime rt = Runtime.getRuntime();
		String svgGraph = null;
		Process p = null;

		try {
			p = rt.exec(new String[] { "dot", "-Tsvg" });
			LOGGER.debug("Found valid Graphviz installation");
			try (Writer stdin = new OutputStreamWriter(p.getOutputStream())) {
				createDotLangGraph(stdin, graph);
				// Hit return
				stdin.append("\n");
			}
			LOGGER.debug("Provided DotLang String to Graphviz");
			try (final BufferedReader br = new BufferedReader(new InputStreamReader(p.getInputStream()))) {
				svgGraph = br.lines().parallel()
						.filter(s -> s.matches("^(?!<polygon\\s.*stroke=\"none\")"
								+ "(<\\/?svg\\W|<\\/?g\\W|<polygon|<ellipse|<text|<path|\\sviewBox)" + ".*$"))
						.map(s -> s
								.replaceFirst("(\\sfont-family=\".*\"\\sfont-size=\".*\\d\"|<title>.*<\\/title>)", "")
								.intern())
						.collect(Collectors.joining("\n"));
			}
			try (BufferedReader br = new BufferedReader(new InputStreamReader(p.getErrorStream()))) {
				final String err = br.lines().collect(Collectors.joining("\n"));
				if (!err.equals("")) {
					LOGGER.debug("Error occured when using Graphviz: {}", err);
					throw new RuntimeException("Graphviz terminated with errors");
				}
			}
			LOGGER.debug("Fetched SVG String from Graphviz");
		} finally {
			if (p != null) {
				p.destroy();
			}
		}

		return svgGraph;
	}

	private static void createDotLangGraph(final Appendable ap, final mxGraph graph) throws IOException {

		final Set<String> subgraphs = new HashSet<>();
		final Collection<Object> allNodes = new LinkedList<>();
		final Deque<Object> stack = new LinkedList<>();
		Map<String, Object> style;
		Object[] nodeSet;
		mxCell cell;
		String id;

		nodeSet = graph.getChildVertices(graph.getDefaultParent());
		stack.addAll(Arrays.asList(nodeSet));
		allNodes.addAll(Arrays.asList(nodeSet));

		// TODO Add app name
		ap.append("digraph G {");
		ap.append(" compound=true;");
		ap.append(" style=filled;");
		ap.append(" node [style=filled,shape=box];");

		// Write all nodes and subgraphs
		while (!stack.isEmpty()) {
			cell = (mxCell) stack.getLast();
			id = cell.getId();
			if (subgraphs.contains(id)) {
				stack.removeLast();
				ap.append(" }");
			} else {
				style = graph.getCellStyle(cell);
				nodeSet = graph.getChildVertices(cell);
				if (nodeSet.length == 0) {
					ap.append(" ").append(id).append(" [label=<").append(makeHTML(graph.getLabel(cell)))
							.append(">,color=\"").append(style.get(mxConstants.STYLE_FILLCOLOR).toString())
							.append("\"];");
					stack.removeLast();
				} else {
					stack.addAll(Arrays.asList(nodeSet));
					allNodes.addAll(Arrays.asList(nodeSet));
					subgraphs.add(id);
					final String label = makeHTML(graph.getLabel(cell));
					ap.append(" subgraph cluster_").append(id).append(" {");
					ap.append(" label=<").append(label).append(">;");
					ap.append(" color=\"").append(style.get(mxConstants.STYLE_FILLCOLOR).toString()).append("\";");
					// Special node for edges pointing to subgraph
					ap.append("\"").append(id)
							.append("_0\" [style=filled,shape=circle,fixedsize=true,width=.25,color=black,label=<>];");
				}
			}
		}

		Object[] edgeSet;
		String tarId;
		for (final Object node : allNodes) {
			edgeSet = graph.getEdges(node, null, false, true, true);
			id = ((mxCell) node).getId();
			for (final Object edge : edgeSet) {
				cell = (mxCell) graph.getOpposites(new Object[] { edge }, node)[0];
				tarId = cell.getId();
				style = graph.getCellStyle(edge);
				ap.append(" ");
				if (subgraphs.contains(id)) {
					ap.append("\"").append(id).append("_0\"");
				} else {
					ap.append(id);
				}
				ap.append(" -> ");
				if (subgraphs.contains(tarId)) {
					ap.append("\"").append(tarId).append("_0\"");
				} else {
					ap.append(tarId);
				}
				ap.append(" [color=\"").append(style.get(mxConstants.STYLE_STROKECOLOR).toString()).append("\",label=<")
						.append(makeHTML(graph.getLabel(edge))).append(">");
				// if (subgraphs.contains(id)) {
				// ap.append(",ltail=cluster_").append(id);
				// }
				// if (subgraphs.contains(tarId)) {
				// ap.append(",lhead=cluster_").append(tarId);
				// }
				ap.append("];");
			}
		}

		ap.append(" }");
	}

	private static String makeHTML(final String str) {
		return str.replace("&", "&amp;").replace("\"", "&quot;").replace("<", "&lt;").replace(">", "&gt;");
	}

	private static String buildGraphWithJGraphX(final mxGraph graph) {

		final mxOrganicLayout ol = new mxOrganicLayout(graph);
		final mxGraphLayout layout = new GroupedLayout(ol);
		layout.execute(graph.getDefaultParent());

		return graphToSVG(graph);
	}

	private static String graphToSVG(final mxGraph graph) {

		Document svgDoc = null;
		try {
			final DocumentBuilderFactory f = DocumentBuilderFactory.newInstance();
			final DocumentBuilder builder = f.newDocumentBuilder();
			svgDoc = builder.newDocument();
			final Element rootElement = svgDoc.createElement("svg");
			svgDoc.appendChild(rootElement);
		} catch (final ParserConfigurationException e) {
			LOGGER.error("Error when creating SVG document: " + e.getMessage());
		}

		final mxSvgCanvas canvas = new mxSvgCanvas(svgDoc);
		graph.drawGraph(canvas);
		String svg = mxXmlUtils.getXml(canvas.getDocument());
		final mxRectangle graphSize = graph.getGraphBounds();

		svg = svg.replaceFirst("<svg>", "<svg height=\"" + (int) Math.ceil(graphSize.getHeight() + SVG_GRAPH_EXTENSION)
				+ "\" width=\"" + (int) Math.ceil(graphSize.getWidth() + SVG_GRAPH_EXTENSION) + "\">");
		return svg;
	}

	private static mxGraph initGraph() {

		final mxGraph graph = new mxGraph();
		graph.setExtendParentsOnAdd(true);
		graph.setAllowDanglingEdges(false);
		graph.setAllowLoops(true);
		graph.setAutoSizeCells(true);

		final mxStylesheet stylesheet = graph.getStylesheet();
		final Hashtable<String, Object> style = new Hashtable<>();
		style.put(mxConstants.STYLE_SHAPE, mxConstants.SHAPE_RECTANGLE);
		style.put(mxConstants.STYLE_STROKEWIDTH, 2);
		style.put(mxConstants.STYLE_FONTCOLOR, "black");
		stylesheet.putCellStyle(GRAPH_STYLE_NAME, style);
		return graph;
	}

	private Object getVertex(final mxGraph graph, final PathElement codeElement) {

		Object vertex;
		if (this.mVertexMap.containsKey(codeElement)) {
			vertex = this.mVertexMap.get(codeElement);
		} else if (codeElement instanceof ResourceElement) {
			final ResourceElement p = (ResourceElement) codeElement;
			vertex = insertVertex(graph, p.getPermissionName(), graph.getDefaultParent(), RESOURCE_COLOR);
			this.mVertexMap.put(p, vertex);
		} else if (codeElement instanceof ClassElement) {
			final ClassElement c = (ClassElement) codeElement;
			vertex = insertVertex(graph, c.getClassName(), graph.getDefaultParent(), CLASS_COLOR);
			this.mVertexMap.put(c, vertex);
		} else if (codeElement instanceof MethodElement) {
			final MethodElement m = (MethodElement) codeElement;
			final Object newParent = getVertex(graph, m.getClassElement());
			vertex = insertVertex(graph, m.getMethodSubSignature(), newParent, METHOD_COLOR);
			this.mVertexMap.put(m, vertex);
		} else if (codeElement instanceof StatementElement) {
			final StatementElement u = (StatementElement) codeElement;
			final Object newParent = getVertex(graph, u.getMethodElement());
			vertex = insertVertex(graph, u.getStatementString(), newParent, STATEMENT_COLOR);
			this.mVertexMap.put(u, vertex);
		} else {
			throw new IllegalArgumentException("Did not expect vertex of type " + codeElement.getClass());
		}
		return vertex;
	}

	private static Object insertVertex(final mxGraph graph, final String label, final Object parent,
			final String color) {
		return graph.insertVertex(parent, null, label, 0, 0, 0, 0,
				GRAPH_STYLE_NAME + ";" + mxConstants.STYLE_FILLCOLOR + "=" + color);
	}

	private static Object insertEdge(final mxGraph graph, final Object from, final Object to, final String color,
			final int index) {
		return graph.insertEdge(graph.getDefaultParent(), null, index, from, to,
				GRAPH_STYLE_NAME + ";" + mxConstants.STYLE_STROKECOLOR + "=" + color);
	}

}
