/**
 *
 */
package de.upb.pga3.panda2.extension.lvl2a.analyzer;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.upb.pga3.panda2.core.datastructures.AnalysisGraph;
import de.upb.pga3.panda2.core.datastructures.AnalysisResult;
import de.upb.pga3.panda2.core.datastructures.EnhancedInput;
import de.upb.pga3.panda2.core.datastructures.Permission;
import de.upb.pga3.panda2.core.services.CoreServices;
import de.upb.pga3.panda2.extension.Analyzer;
import de.upb.pga3.panda2.extension.lvl2a.AnalysisGraphLvl2a;
import de.upb.pga3.panda2.extension.lvl2a.AnalysisResultLvl2a;
import de.upb.pga3.panda2.extension.lvl2a.ComparisonAnalysisResultLvl2a;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.FlowPath;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.ResourceElement;
import soot.Unit;
import soot.toolkits.scalar.Pair;

/**
 * Analyzer for level 2a
 *
 * @author nptsy
 * @author monika
 * @author Fabian
 */
public class AnalyzerLvl2a implements Analyzer {

	private final SourceAndSinkComputer mSusiComputer;

	private final BackwardSlicer mBackwardSlicer;

	private final PathFinder mPathFinder;

	private final ResultComparerLvl2a mResComparer;

	private static final Logger LOGGER = LogManager.getLogger(AnalyzerLvl2a.class);

	/**
	 * Constructor
	 */
	public AnalyzerLvl2a() {
		this.mSusiComputer = new SourceAndSinkComputer();
		this.mBackwardSlicer = new BackwardSlicer();
		this.mPathFinder = new PathFinder();
		this.mResComparer = new ResultComparerLvl2a();
	}

	/**
	 * compute sources and sinks on graph
	 *
	 * @param inGraph
	 *            the AnalysisGraph of lvl2a
	 */

	private Pair<Map<Permission, List<Unit>>, Map<Permission, List<Unit>>> computeSourcesAndSinks(
			final AnalysisGraphLvl2a inGraph) {
		/*
		 * get list of nodes of the analysisGraph which are in type Unit. we
		 * don't consider the node [method]
		 */
		LOGGER.info("AnalyzerLvl2a starts computing sources and sinks");

		final Object[] lstNode = inGraph.getNodes().toArray();
		final List<Unit> lstNodeUnit = new ArrayList<>();
		if (lstNode != null && lstNode.length > 0) {
			for (final Object obj : lstNode) {
				if (obj instanceof Unit) {
					lstNodeUnit.add((Unit) obj);
				}
			}
		}

		// process for sources
		final Map<Permission, List<Unit>> sources = this.mSusiComputer.computeSources(lstNodeUnit,
				(EnhancedInput) inGraph.getInput());
		// process for sinks
		final Map<Permission, List<Unit>> sinks = this.mSusiComputer.computeSinks(lstNodeUnit,
				(EnhancedInput) inGraph.getInput());

		LOGGER.info("AnalyzerLvl2a finished computing sources and sinks");

		return new Pair<>(sources, sinks);
	}

	/**
	 * extract execution paths of graph
	 *
	 * @param inSlicedGraph
	 *            the analysis graph of lvl2a
	 *
	 */
	private void extractExecutionPaths(final Map<Permission, List<Unit>> inSlicedSources, final Permission inPermission,
			final Unit inSink, final AnalysisResultLvl2a inRes) {

		// add processing here
		final Set<Permission> setKeys = inSlicedSources.keySet();
		for (final Permission perm : setKeys) {
			final List<Unit> sources = inSlicedSources.get(perm);
			for (final Unit source : sources) {
				final Collection<FlowPath> lstPaths = this.mPathFinder.findPaths(
						new Pair<Permission, Object>(perm, source), new Pair<Permission, Object>(inPermission, inSink));

				if (lstPaths != null && !lstPaths.isEmpty()) {
					// Store path in result
					for (final FlowPath path : lstPaths) {
						inRes.addPath(path);
					}
				}
			}
		}
	}

	@Override
	public AnalysisResult analyze(final AnalysisGraph analgraph, final AnalysisResult prevRes) {
		LOGGER.info("Analyzer started (Source&Sink - Intra App - Level 2a).");

		// GraphPlotter.getInstance().resetAndPlotGraph((AnalysisGraphLvl2a)
		// analgraph);

		AnalysisResultLvl2a res = null;
		if (analgraph instanceof AnalysisGraphLvl2a) {
			final AnalysisGraphLvl2a ag2a = (AnalysisGraphLvl2a) analgraph;
			if (prevRes == null) {
				res = performSummaryAnalysis(ag2a);
			} else {
				if (prevRes instanceof AnalysisResultLvl2a) {
					res = performComparisonAnalysis(ag2a, (AnalysisResultLvl2a) prevRes);
				} else {
					throw new IllegalArgumentException("Previous result is not of type Level2a!");
				}
			}
		} else {
			throw new IllegalArgumentException("Graph is not of type Level2a!");
		}
		res.createFilters();

		LOGGER.info("Analyzer finished (Source&Sink - Intra App - Level 2a)");
		return res;
	}

	private AnalysisResultLvl2a performSummaryAnalysis(final AnalysisGraphLvl2a angraph) {

		// get the app name as well as app version
		final EnhancedInput input = (EnhancedInput) angraph.getInput();
		final String appName = input.getAppName();

		// compute source and sink of analysis graph
		final AnalysisResultLvl2a res = new AnalysisResultLvl2a();
		res.setAppName(appName);
		res.initialize(CoreServices.getXMLParserInstance());
		final Pair<Map<Permission, List<Unit>>, Map<Permission, List<Unit>>> pairSouSi = computeSourcesAndSinks(
				angraph);
		final Map<Permission, List<Unit>> sources = pairSouSi.getO1();
		final Map<Permission, List<Unit>> sinks = pairSouSi.getO2();

		// for each sink, extract path to it
		this.mPathFinder.setGraph(angraph);

		// add list of sources and sinks to the AnalysisResultLvl2a
		res.addSinks(sinks.keySet().stream().map(p -> ResourceElement.get(p)).collect(Collectors.toSet()));
		res.addSources(sources.keySet().stream().map(p -> ResourceElement.get(p)).collect(Collectors.toSet()));

		LOGGER.info("BackwardSlicer and PathFinder start computing");

		LOGGER.debug("Found {} sink permissions", sinks.size());
		for (final Map.Entry<Permission, List<Unit>> entry : sinks.entrySet()) {
			// permission
			final Permission perm = entry.getKey();
			// list of sink
			final List<Unit> lstSinks = entry.getValue();
			LOGGER.debug("Found {} statements for sink {}", lstSinks.size(), perm.getName());
			for (final Unit sink : lstSinks) {
				final Map<Permission, List<Unit>> mapPermSources = this.mBackwardSlicer.slice(angraph, sink, sources);
				if (mapPermSources != null && !mapPermSources.isEmpty()) {
					extractExecutionPaths(mapPermSources, perm, sink, res);
				}
			}
		}
		LOGGER.info("BackwardSlicer and PathFinder finished computing");

		return res;
	}

	private ComparisonAnalysisResultLvl2a performComparisonAnalysis(final AnalysisGraphLvl2a angraph,
			final AnalysisResultLvl2a prevRes) {

		final AnalysisResultLvl2a currRes = performSummaryAnalysis(angraph);
		return this.mResComparer.comparePrevAndCur(prevRes, currRes);
	}
}
