package de.upb.pga3.panda2.extension.lvl2a.analyzer;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.upb.pga3.panda2.core.datastructures.EnhancedInput;
import de.upb.pga3.panda2.core.datastructures.Permission;
import de.upb.pga3.panda2.core.datastructures.Transition;
import de.upb.pga3.panda2.extension.lvl2a.AnalysisGraphLvl2a;
import de.upb.pga3.panda2.extension.lvl2a.ParameterNode;
import de.upb.pga3.panda2.extension.lvl2a.TransitionLvl2a;
import de.upb.pga3.panda2.extension.lvl2a.TransitionType;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.ClassElement;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.FlowPath;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.MethodElement;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.PathElement;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.ResourceElement;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.StatementElement;
import soot.SootClass;
import soot.SootMethod;
import soot.Unit;
import soot.toolkits.scalar.Pair;

/**
 *
 * @author Fabian
 * @author nptsy
 */
public class PathFinder {
	private static final Logger LOGGER = LogManager.getLogger(PathFinder.class);

	private AnalysisGraphLvl2a mGraph;

	public void setGraph(final AnalysisGraphLvl2a inGraph) {

		if (inGraph != null) {
			this.mGraph = inGraph;
		} else {
			throw new IllegalArgumentException("The given graph must not be null!");
		}
	}

	public Collection<FlowPath> findPaths(final Pair<Permission, Object> inSource,
			final Pair<Permission, Object> inSink) {

		final EnhancedInput input = (EnhancedInput) this.mGraph.getInput();

		final Collection<FlowPath> lstFlowPaths = new LinkedHashSet<>();
		final TransitionLvl2a rootNode = new TransitionLvl2a(inSource.getO1(), inSource.getO2());
		try {
			final List<List<TransitionLvl2a>> transList = discoverPathInGraph(rootNode, inSink.getO2());
			for (final List<TransitionLvl2a> lstTrans : transList) {
				try {
					if (!lstTrans.isEmpty()) {
						final FlowPath path = new FlowPath(lstTrans.size() + 2);
						final Iterator<TransitionLvl2a> transit = lstTrans.listIterator();
						TransitionLvl2a trans = null;
						while (transit.hasNext()) {
							trans = transit.next();
							addElemToPath(path, trans.getSource(), input);
						}
						addElemToPath(path, trans.getTarget(), input);
						addElemToPath(path, inSink.getO1(), input);
						lstFlowPaths.add(path);
					}
				} catch (final Exception ex) {
					LOGGER.warn("Error: " + ex.getMessage());
				}
			}

		} catch (final Exception ex) {
			LOGGER.warn("Cannot find paths from " + inSource.getO2().toString() + " to " + inSink.getO2().toString());
			LOGGER.error("Reason: " + ex.getMessage());
			LOGGER.debug(ex);
			return null;
		}

		LOGGER.debug("Found {} path between source {} and sink {}", lstFlowPaths.size(), inSource.getO1().getName(),
				inSink.getO1().getName());

		return lstFlowPaths;
	}

	private static void addElemToPath(final FlowPath path, final Object node, final EnhancedInput input) {

		PathElement elem;
		elem = toPathElem(node, input);
		if (elem != null) {
			path.addElement(elem);
		}
	}

	private static PathElement toPathElem(final Object node, final EnhancedInput input) {

		if (node instanceof Permission) {
			return ResourceElement.get((Permission) node);
		} else if (node instanceof SootClass) {
			return ClassElement.get((SootClass) node, input);
		} else if (node instanceof SootMethod) {
			return MethodElement.get((SootMethod) node, input);
		} else if (node instanceof Unit) {
			return StatementElement.get((Unit) node, input);
		}
		// Skip parameter nodes
		return null;
	}

	private List<List<TransitionLvl2a>> discoverPathInGraph(final TransitionLvl2a inSourceTrans, final Object inSink) {

		// Perform DFS on graph to find the path
		final List<TransitionLvl2a> path = new ArrayList<>();
		final List<List<TransitionLvl2a>> lstPaths = new ArrayList<>();
		final Collection<TransitionLvl2a> lstVisitedTrans = new HashSet<>();
		recDiscoverPath(path, inSourceTrans, inSink, lstPaths, lstVisitedTrans);
		if (!lstPaths.isEmpty()) {
			return lstPaths;
		} else {
			throw new IllegalStateException("A path between source and sink could not be found!");
		}
	}

	private void recDiscoverPath(final List<TransitionLvl2a> inPath, final TransitionLvl2a inCurrentTrans,
			final Object inSinkObj, final List<List<TransitionLvl2a>> inListPaths,
			final Collection<TransitionLvl2a> inListVisitedTrans) {

		if (inListVisitedTrans.contains(inCurrentTrans)) {
			return;
		} else {
			inListVisitedTrans.add(inCurrentTrans);
		}
		final Object sourceTargetObj = inCurrentTrans.getTarget();

		inPath.add(inCurrentTrans);

		if (sourceTargetObj.equals(inSinkObj)) {
			inListPaths.add(new ArrayList<>(inPath));
			inPath.remove(inCurrentTrans);
			inListVisitedTrans.remove(inCurrentTrans);
			return;
		}

		final Set<TransitionLvl2a> transOut = filterTrans(sourceTargetObj);
		for (final TransitionLvl2a tran : transOut) {
			// TODO Check if that is necessary
			if (!inPath.contains(tran)) {
				recDiscoverPath(inPath, tran, inSinkObj, inListPaths, inListVisitedTrans);
			}
		}
		inListVisitedTrans.remove(inCurrentTrans);
		inPath.remove(inCurrentTrans);
	}

	private Set<TransitionLvl2a> filterTrans(final Object sourceTargetObj) {

		final Set<Transition> inTrans = this.mGraph.getOutgoingTransitions(sourceTargetObj);
		TransitionLvl2a t2a = null;
		final Set<TransitionLvl2a> filteredTrans = new HashSet<>();

		for (final Transition t : inTrans) {
			if (t instanceof TransitionLvl2a) {
				t2a = (TransitionLvl2a) t;
				final TransitionType type = t2a.getTransitionType();
				final Object target = t2a.getTarget();
				if (type.equals(TransitionType.CONTROLDEPENDENCY) && target instanceof ParameterNode) {
					continue;
				}
				if (type.equals(TransitionType.CONTROLFLOW)) {
					continue;
				}
				if (type.equals(TransitionType.SUMMARY)) {
					continue;
				}
				filteredTrans.add(t2a);
			}
		}
		return filteredTrans;
	}
}
