/**
 *
 */
package de.upb.pga3.panda2.extension.lvl2a.graphgenerator;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.upb.pga3.panda2.core.SootAdapter;
import de.upb.pga3.panda2.core.SootAdapter.SootPhase;
import de.upb.pga3.panda2.core.datastructures.EnhancedInput;
import de.upb.pga3.panda2.core.datastructures.Message;
import de.upb.pga3.panda2.core.datastructures.MessageType;
import de.upb.pga3.panda2.core.datastructures.Transition;
import de.upb.pga3.panda2.core.services.IntentInformation;
import de.upb.pga3.panda2.extension.lvl2a.AnalysisGraphLvl2a;
import de.upb.pga3.panda2.extension.lvl2a.ParamType;
import de.upb.pga3.panda2.extension.lvl2a.ParameterNode;
import de.upb.pga3.panda2.extension.lvl2a.TransitionLvl2a;
import de.upb.pga3.panda2.extension.lvl2a.TransitionType;
import de.upb.pga3.panda2.utilities.Constants;
import gnu.trove.map.TIntObjectMap;
import gnu.trove.map.TObjectIntMap;
import gnu.trove.map.hash.TIntObjectHashMap;
import gnu.trove.map.hash.TObjectIntHashMap;
import gnu.trove.set.TIntSet;
import gnu.trove.set.hash.TIntHashSet;
import soot.Body;
import soot.Local;
import soot.PatchingChain;
import soot.Scene;
import soot.SootClass;
import soot.SootMethod;
import soot.Transform;
import soot.Unit;
import soot.Value;
import soot.ValueBox;
import soot.jimple.AssignStmt;
import soot.jimple.InvokeExpr;
import soot.jimple.ReturnStmt;
import soot.jimple.Stmt;
import soot.jimple.toolkits.callgraph.CallGraph;
import soot.jimple.toolkits.callgraph.Targets;
import soot.toolkits.graph.ExceptionalUnitGraph;
import soot.toolkits.graph.UnitGraph;
import soot.util.Chain;

/**
 *
 * @author nptsy
 * @author Fabian
 * @author Monika
 * @author Ram
 */
public class NodeLinkerLvl2a {

	private static final Logger LOGGER = LogManager.getLogger(NodeLinkerLvl2a.class);

	// analysis graph of lvl 2a
	private AnalysisGraphLvl2a mAnalysisGraph;

	// Method - Local Variable - Statement map
	// private Map<SootMethod, Map<Local, Unit>> methodLocDefStmtMap = new
	// HashMap<>();
	private Map<SootMethod, Map<Unit, List<Unit>>> useStmtDefStmtMap = new HashMap<>();
	private Map<SootMethod, List<Unit>> methodInvkStmtMap = new HashMap<>();
	private Map<SootMethod, List<Value>> methodParamMap = new HashMap<>();
	private Map<SootMethod, Map<Value, Unit>> methodParamIdentityUnitMap = new HashMap<>();
	private Map<SootMethod, Unit> methodRetnStmtMap = new HashMap<>();

	private List<Transition> intentDataTransitions = new ArrayList<>();
	/**
	 * map that contains pairs of [key, value]. Key is signature of method and
	 * value is the unit graph of the method
	 */
	private Map<SootMethod, UnitGraph> mMapUnitGraph;

	/**
	 * enriches the given analysis graph with transitions of different types
	 *
	 * @param inGraph
	 *            analysis graph of type AnalysisGraphLvl2a
	 * @return the analysis graph enriched with transitions and parameter nodes
	 */
	public AnalysisGraphLvl2a link(final AnalysisGraphLvl2a inGraph) {
		this.mAnalysisGraph = inGraph;

		getGraphsFromSoot();
		createControlFlowGraph();
		modelControlDependency();
		addDataEdges();
		addParamEdges();
		addParamDataFlow();
		addSummaryEdges();
		addMethodCallEdges();
		computeAndAddDataEdgesForIntents();

		return this.mAnalysisGraph;
	}

	/**
	 * get all kinds of graphs in soot
	 */
	private void getGraphsFromSoot() {

		// create dummy main method for SOOT
		final DummyMain dummyMain = new DummyMain();
		SootAdapter.getInstance().setEntryPoint(dummyMain.initialize());

		// create body transformer for unit graph
		final NodeLinkTransformerPart2 bodyTransformer = new NodeLinkTransformerPart2();
		SootAdapter.getInstance().addTransformer(new Transform("jtp.lvl2ABodyTransform", bodyTransformer),
				SootPhase.JTP);

		SootAdapter.getInstance().run(SootPhase.JTP);

		// Initialise the Mappings obtained from Soot BodyTransformer. Mappings
		// to be used for DataFlow and Parameter Transition Modelling//

		this.mMapUnitGraph = bodyTransformer.getMap();
		// this.methodLocDefStmtMap = bodyTransformer.getLocDefStmtMap();
		this.useStmtDefStmtMap = bodyTransformer.getUseStmtDefStmtMap();
		this.methodInvkStmtMap = bodyTransformer.getMethodInvkExprMap();
		this.methodParamMap = bodyTransformer.getMethodParamMap();
		this.methodParamIdentityUnitMap = bodyTransformer.getMethodParamIdentityUnitMap();
		this.methodRetnStmtMap = bodyTransformer.getMethodRetnStmtMap();
	}

	/**
	 * create control flow graph
	 *
	 */
	private void createControlFlowGraph() {

		// call graph of soot
		final CallGraph callgraph = Scene.v().getCallGraph();

		// get list of all application classes inside the android application
		final Chain<SootClass> classList = Scene.v().getApplicationClasses();

		// list of

		// get list of entry methods
		final List<SootMethod> lstEntries = Scene.v().getEntryPoints();

		if (lstEntries != null && !lstEntries.isEmpty()) {

			final SootMethod dummyMain = lstEntries.get(0);
			/*
			 * -----------------------------------------------------------------
			 * process for dummy main method and others
			 * -----------------------------------------------------------------
			 */

			Iterator targets = new Targets(callgraph.edgesOutOf(dummyMain));
			while (targets.hasNext()) {
				final SootMethod targetMethod = (SootMethod) targets.next();
				final TransitionLvl2a trans = createTransition(dummyMain, targetMethod, TransitionType.CONTROLFLOW);

				// add transition to the analysis graph
				this.mAnalysisGraph.addTransition(trans);
			}

			/*
			 * -----------------------------------------------------------------
			 * process for app methods and others
			 * -----------------------------------------------------------------
			 */
			for (final SootClass sootClass : classList) {
				// class
				// G.v().out.println("======== Class: " + sootClass.getName());

				// get list methods of the current class.
				final List<SootMethod> methodList = sootClass.getMethods();

				for (final SootMethod sm : methodList) {
					// method
					// G.v().out.println("Method: " + sm.getName());

					// get list of out going edge from the method
					targets = new Targets(callgraph.edgesOutOf(sm));
					while (targets.hasNext()) {
						final SootMethod targetMethod = (SootMethod) targets.next();
						final TransitionLvl2a trans = createTransition(sm, targetMethod, TransitionType.CONTROLFLOW);

						// add transition to the analysis graph
						this.mAnalysisGraph.addTransition(trans);
					}

					/*
					 * ---------------------------------------------------------
					 * processing for body of method below
					 * ---------------------------------------------------------
					 */
					if (!sm.isAbstract() && !sm.getDeclaringClass().isInterface()) {

						boolean isFirstStatement = true;
						try {

							// get body <= this is not official way for getting
							// body
							// of a method
							final Body body = sm.retrieveActiveBody();
							final PatchingChain<Unit> chainUnits = body.getUnits();

							// get unit graph of the body
							UnitGraph graph;

							// in case map of unit graph is not null
							if (this.mMapUnitGraph != null) {
								graph = this.mMapUnitGraph.get(sm);

								// graph is not null
								if (graph == null) {
									graph = new ExceptionalUnitGraph(body);
								}
							} // in case map of unit graph is null
							else {
								graph = new ExceptionalUnitGraph(body);
							}

							// get list units of the body
							for (final Unit unit : chainUnits) {

								/*
								 * add connection between method and the first
								 * statement
								 */
								if (isFirstStatement) {
									isFirstStatement = false;
									final TransitionLvl2a transUnit = createTransition(sm, unit,
											TransitionType.CONTROLFLOW);

									this.mAnalysisGraph.addTransition(transUnit);
								}

								/*
								 * add connection between statements and
								 * statements (units and units)
								 */
								final List<Unit> lstUnits = graph.getSuccsOf(unit);
								// get successor of current unit
								for (final Unit preds : lstUnits) {
									final TransitionLvl2a transUnit = createTransition(unit, preds,
											TransitionType.CONTROLFLOW);

									this.mAnalysisGraph.addTransition(transUnit);
								}
							}

						} catch (final Exception ex) {
							LOGGER.warn("The method " + sm.getSubSignature() + " has no body. Exception: "
									+ ex.getMessage());
						}
					}
				}
			}
		}
	}

	/**
	 * Adding Data dependencies for the Local variables.
	 */

	private void addDataEdges() {

		for (final Entry<SootMethod, Map<Unit, List<Unit>>> methodToUseDefMap : this.useStmtDefStmtMap.entrySet()) {

			for (final Entry<Unit, List<Unit>> useToDefsMap : methodToUseDefMap.getValue().entrySet()) {

				if (!useToDefsMap.getValue().isEmpty()) {

					for (final Unit defUnit : useToDefsMap.getValue()) {

						final TransitionLvl2a dataTrans = createTransition(defUnit, useToDefsMap.getKey(),
								TransitionType.DATAFLOW);
						this.mAnalysisGraph.addTransition(dataTrans);
					}
				}
			}
		}

	}

	/**
	 * Add Parameter Nodes and their associated Parameter, DataFlow and Control
	 * Dependence transitions to analysis graph
	 */

	private void addParamEdges() {

		// create FORMAL_IN parameter nodes for every method
		for (final SootClass c : ((EnhancedInput) this.mAnalysisGraph.getInput()).getAppClasses()) {
			for (final SootMethod method : c.getMethods()) {
				if (this.methodParamMap.containsKey(method)) {
					for (final Value param : this.methodParamMap.get(method)) {

						final int index = this.methodParamMap.get(method).indexOf(param);
						final ParameterNode destNode = new ParameterNode(param, ParamType.FORMAL_IN, method, index);
						this.mAnalysisGraph.addParameterNode(destNode);
						// create control dependency edge from method call to
						// formal-in node
						final TransitionLvl2a cntlDepTrans1 = createTransition(method, destNode,
								TransitionType.CONTROLDEPENDENCY);
						this.mAnalysisGraph.addTransition(cntlDepTrans1);
					}
				}

				// create FORMAL_OUT Parameter Node.
				if (this.methodRetnStmtMap.containsKey(method)) {
					final Unit retUnit = this.methodRetnStmtMap.get(method);
					final ReturnStmt rSt = (ReturnStmt) retUnit;
					final Value srcVal = rSt.getOp();

					final ParameterNode srcNode = new ParameterNode(srcVal, ParamType.FORMAL_OUT, method, 0);
					this.mAnalysisGraph.addParameterNode(srcNode);

					// create control dependency edge from method call to
					// formal-out node
					final TransitionLvl2a cntlDepTrans2 = createTransition(method, srcNode,
							TransitionType.CONTROLDEPENDENCY);
					this.mAnalysisGraph.addTransition(cntlDepTrans2);
				}

			}
		}

		// Modelling Parameter Transitions for the Invoke Units in the program.
		final Set<SootMethod> methods = this.methodInvkStmtMap.keySet();
		for (final SootMethod sm : methods) {
			final List<Unit> invkUnits = this.methodInvkStmtMap.get(sm);
			for (final Unit invkUnit : invkUnits) {
				final Stmt st = (Stmt) invkUnit;
				final InvokeExpr iExpr = st.getInvokeExpr();
				final SootMethod invkdMethod = iExpr.getMethod();
				if (this.methodParamMap.keySet().contains(invkdMethod)) {
					final List<Value> args = iExpr.getArgs();
					final List<Value> parameters = this.methodParamMap.get(invkdMethod);
					for (int i = 0; i < args.size(); i++) {
						final Value src = args.get(i);
						final Value dest = parameters.get(i);

						// Create ACTUAL_IN parameter Nodes for an Invoke Unit.
						final ParameterNode actualInNode = new ParameterNode(src, ParamType.ACTUAL_IN, invkUnit, i);
						this.mAnalysisGraph.addParameterNode(actualInNode);
						// create control dependency edge from method call to
						// actual-in node
						final TransitionLvl2a cntlDepTrans = createTransition(invkUnit, actualInNode,
								TransitionType.CONTROLDEPENDENCY);
						this.mAnalysisGraph.addTransition(cntlDepTrans);

						// create param-in transition from
						// actual-in to formal-in node
						if (this.mAnalysisGraph.hasParameterNodes(invkdMethod)) {
							final Set<ParameterNode> params = this.mAnalysisGraph.getParameterNodes(invkdMethod);
							for (final ParameterNode param : params) {
								if (param.getIndex() == i && param.getType() == ParamType.FORMAL_IN) {
									final TransitionLvl2a paramIn = createTransition(actualInNode, param,
											TransitionType.PARAMIN);
									this.mAnalysisGraph.addTransition(paramIn);
								}
							}

						}
					}
				}
				// If the Invoked Method returns a Value, FORMAL_OUT and
				// ACTUAL_OUT nodes to be modelled.

				if (st instanceof AssignStmt && this.methodRetnStmtMap.containsKey(invkdMethod)) {
					final AssignStmt aSt = (AssignStmt) st;
					final Value destVal = aSt.getLeftOp();
					final Unit retUnit = this.methodRetnStmtMap.get(invkdMethod);
					final ReturnStmt rSt = (ReturnStmt) retUnit;
					final Value srcVal = rSt.getOp();

					// Create ACTUAL_OUT Parameter Node.
					final ParameterNode actualOutNode = new ParameterNode(destVal, ParamType.ACTUAL_OUT, invkUnit, 0);
					this.mAnalysisGraph.addParameterNode(actualOutNode);

					final TransitionLvl2a cntlDepTrans3 = createTransition(invkUnit, actualOutNode,
							TransitionType.CONTROLDEPENDENCY);
					this.mAnalysisGraph.addTransition(cntlDepTrans3);

					// create param-out transition from
					// formal-out to actual-out node

					if (this.mAnalysisGraph.hasParameterNodes(invkdMethod)) {
						final Set<ParameterNode> params = this.mAnalysisGraph.getParameterNodes(invkdMethod);
						for (final ParameterNode param : params) {
							if (param.getParameterValue().equals(srcVal)) {
								if (param.getType() == ParamType.FORMAL_OUT) {
									final TransitionLvl2a paramOut = createTransition(param, actualOutNode,
											TransitionType.PARAMOUT);
									this.mAnalysisGraph.addTransition(paramOut);

								}
							}
						}
					}
				}

			}
		}
	}

	/**
	 * adds data flow edges and parameter edges related to the parameter nodes
	 * to the analysis graph
	 */
	private void addParamDataFlow() {
		for (final Unit invkUnit : this.mAnalysisGraph.getParameterizedInvokeStmnts()) {
			for (final ParameterNode param : this.mAnalysisGraph.getParameterNodes(invkUnit)) {

				if (param.getType() == ParamType.ACTUAL_IN) {
					// Add DataFlow transition between ACTUAL_IN nodes and
					// Units that define the Argument Values.
					// Source Node - Definition Unit,
					// Destination Node - ACTUAL_IN Parameter Node

					final EnhancedInput enhancedInput = (EnhancedInput) this.mAnalysisGraph.getInput();
					final Body body = enhancedInput.getBodyForUnit(invkUnit);

					final List<Unit> defUnits = this.useStmtDefStmtMap.get(body.getMethod()).get(invkUnit);
					for (final Unit defUnit : defUnits) {
						final List<ValueBox> vBoxes = defUnit.getDefBoxes();
						for (final ValueBox vb : vBoxes) {
							if (vb.getValue().equals(param.getParameterValue())) {
								final TransitionLvl2a dataTransActIn = createTransition(defUnit, param,
										TransitionType.DATAFLOW);
								this.mAnalysisGraph.addTransition(dataTransActIn);
							}
						}
					}
				} else if (param.getType() == ParamType.ACTUAL_OUT) {
					// Adding dataflow transition between ACTUAL_OUT node and
					// next use of the variable.

					final Unit invokeUnit = param.getCorrespondingUnit();
					final List<Transition> transitionsToDelete = new ArrayList();
					for (final Transition t : this.mAnalysisGraph.getOutgoingTransitions(invokeUnit)) {
						final TransitionLvl2a transition = (TransitionLvl2a) t;
						if (transition.getTransitionType() == TransitionType.DATAFLOW) {
							final TransitionLvl2a dataFlow = createTransition(param, transition.getTarget(),
									TransitionType.DATAFLOW);
							this.mAnalysisGraph.addTransition(dataFlow);

							// delete dataflow transition from invokestatement
							// bec it is represented by the new transition now
							transitionsToDelete.add(t);
						}
					}
					for (final Transition t : transitionsToDelete) {
						this.mAnalysisGraph.removeTransition(t);
					}
				} else {
					throw new IllegalStateException("Unexpected param type: " + param.getType());
				}

			}
		}

		for (final SootMethod method : this.mAnalysisGraph.getParameterizedMethods()) {
			for (final ParameterNode param : this.mAnalysisGraph.getParameterNodes(method)) {

				if (param.getType() == ParamType.FORMAL_IN) {

					// Add DataFlow transition between
					// FORMAL_IN nodes and Units that define
					// the Parameter Values.
					// Source Node - FORMAL_IN node,
					// Destination Node - Definition Unit
					// for the Parameter Local

					final Unit idtyUnit = this.methodParamIdentityUnitMap.get(method).get(param.getParameterValue());
					final TransitionLvl2a dataTransFormIn = createTransition(param, idtyUnit, TransitionType.DATAFLOW);
					this.mAnalysisGraph.addTransition(dataTransFormIn);
				} else if (param.getType() == ParamType.FORMAL_OUT) {
					// Adding Dataflow transition between
					// FORMAL_OUT node and the associated Return
					// Unit in the invoked method.
					// Src - Return Unit, Dest - FORMAL_OUT node

					final TransitionLvl2a dataTransFormOut = createTransition(this.methodRetnStmtMap.get(method), param,
							TransitionType.DATAFLOW);
					this.mAnalysisGraph.addTransition(dataTransFormOut);
				} else {
					throw new IllegalStateException("Unexpected param type: " + param.getType());
				}

			}
		}

	}

	/**
	 * adds control dependency edges to the analysis graph
	 */
	private void modelControlDependency() {

		// create reverse control flow graph
		final AnalysisGraphLvl2a reverseCFG = new AnalysisGraphLvl2a(this.mAnalysisGraph.getInput());
		final Set<Object> nodes = this.mAnalysisGraph.getNodes();
		for (final Object node : nodes) {
			final Set<Transition> oldtransitions = this.mAnalysisGraph.getOutgoingTransitions(node);
			for (final Transition t : oldtransitions) {
				if (t instanceof TransitionLvl2a) {
					final TransitionLvl2a oldTransition = (TransitionLvl2a) t;
					if (oldTransition.getTransitionType() == TransitionType.CONTROLFLOW) {
						if (!(oldTransition.getSource() instanceof SootMethod
								&& oldTransition.getTarget() instanceof SootMethod)) {
							final TransitionLvl2a reverseTransition = new TransitionLvl2a(oldTransition.getTarget(),
									oldTransition.getSource(), TransitionType.CONTROLFLOW);
							reverseCFG.addTransition(reverseTransition);
						}
					}
				}
			}
		}

		// add unique entry and exit node
		final Object start = "Start";
		final Object end = "Stop";
		final List<Object> heads = new ArrayList<>();
		final List<Object> tails = new ArrayList<>();
		for (final Object o : nodes) {
			if (reverseCFG.getIncomingTransitions(o).isEmpty()) {
				heads.add(o);
			}
			if (reverseCFG.getOutgoingTransitions(o).isEmpty()) {
				tails.add(o);
			}
		}
		for (final Object object : heads) {
			final TransitionLvl2a trans = new TransitionLvl2a(start, object, TransitionType.CONTROLFLOW);
			reverseCFG.addTransition(trans);
		}
		for (final Object object : tails) {
			final TransitionLvl2a t = new TransitionLvl2a(object, end, TransitionType.CONTROLFLOW);
			reverseCFG.addTransition(t);
		}

		// transform reverse cfg to int-representation
		final TObjectIntMap<Object> mMapNodeToInt = new TObjectIntHashMap<>();
		final TIntObjectMap<Object> mMapIntToNode = new TIntObjectHashMap<>();
		int i = 1;
		for (final Object node : reverseCFG.getNodes()) {
			mMapNodeToInt.put(node, i);
			mMapIntToNode.put(i, node);
			i++;
		}

		// represent transitions in succ-array
		final TIntSet[] succ = new TIntSet[i];
		for (int j = 0; j < i; j++) {
			succ[j] = new TIntHashSet();
		}
		for (final Object node : reverseCFG.getNodes()) {
			final Set<Transition> Transitions = reverseCFG.getOutgoingTransitions(node);
			for (final Transition transition : Transitions) {
				if (transition instanceof TransitionLvl2a) {
					final TransitionLvl2a trans = (TransitionLvl2a) transition;
					if (trans.getTransitionType() == TransitionType.CONTROLFLOW) {
						succ[mMapNodeToInt.get(trans.getSource())].add(mMapNodeToInt.get(trans.getTarget()));
					}
				}
			}
		}

		// compute post-dominators
		final DominatorComputer dc = new DominatorComputer();
		final int[] dom = dc.computeDominatorTree(succ, mMapNodeToInt.get(start), i);

		// get all control flow edges where the target does not post-dominate
		// the source
		final List<TransitionLvl2a> notDominated = new ArrayList<>();
		for (final Object o : nodes) {
			for (final Transition t : this.mAnalysisGraph.getOutgoingTransitions(o)) {
				if (t instanceof TransitionLvl2a) {
					final TransitionLvl2a transition = (TransitionLvl2a) t;
					if (transition.getTransitionType() == TransitionType.CONTROLFLOW) {
						if (!(transition.getSource() instanceof SootMethod
								&& transition.getTarget() instanceof SootMethod)) {
							final int a = mMapNodeToInt.get(transition.getSource());
							final int b = mMapNodeToInt.get(transition.getTarget());
							if (!bDominatesA(a, b, dom)) {
								notDominated.add(transition);
							}
						}
					}
				}
			}
		}
		// list is needed to add dependency edge from start to all nodes that
		// are not dependent on any other node
		final List<Object> allNodes = new ArrayList<>(nodes);

		for (final TransitionLvl2a t : notDominated) {

			final Object x = t.getSource();
			final Object y = t.getTarget();

			final int parentOfX = dom[mMapNodeToInt.get(x)];
			int node = mMapNodeToInt.get(y);
			while (node != 0) {
				node = dom[node];
				if (node != parentOfX && node != mMapNodeToInt.get(x) && node != 0) {
					// control dependent

					final TransitionLvl2a controlDependency = createTransition(x, mMapIntToNode.get(node),
							TransitionType.CONTROLDEPENDENCY);
					this.mAnalysisGraph.addTransition(controlDependency);

					allNodes.remove(mMapIntToNode.get(node));

				} else {
					break;
				}
			}
		}

		// add dependency edge from method to all nodes in that method that are
		// not dependent on any other node

		for (final Object node : allNodes) {
			if (node instanceof Unit) {
				final EnhancedInput ei = (EnhancedInput) this.mAnalysisGraph.getInput();
				final Body body = ei.getBodyForUnit((Unit) node);
				final SootMethod method = body.getMethod();
				final TransitionLvl2a transition = createTransition(method, node, TransitionType.CONTROLDEPENDENCY);
				this.mAnalysisGraph.addTransition(transition);
			}
		}
	}

	/**
	 * Checks whether node b dominates node a in the dominator tree
	 *
	 * @param a
	 *            node in the dominator tree
	 * @param b
	 *            node in the dominator tree
	 * @param dom
	 *            the dominator tree
	 * @return true if b dominates a, false otherwise
	 */
	private boolean bDominatesA(final int a, final int b, final int[] dom) {
		if (dom[a] == 0) {
			return false;
		}
		if (dom[a] == b) {
			return true;
		}
		return bDominatesA(dom[a], b, dom);
	}

	/**
	 * create transition from source element to target element
	 *
	 * @param inEleFrom
	 *            source element
	 * @param inEleTo
	 *            sink element
	 * @param inType
	 *            type of transition
	 * @return an object TransitionLvl2a
	 */
	private TransitionLvl2a createTransition(final Object inEleFrom, final Object inEleTo,
			final TransitionType inType) {
		final TransitionLvl2a trans = new TransitionLvl2a(inEleFrom, inEleTo, inType);

		return trans;
	}

	/**
	 * add transition for explicit intents
	 *
	 * @param inIntent
	 *            a transition of lvl2a
	 * @param inGraph
	 *            AnalysiGraphLvl2a graph for lvl2a
	 */
	public void addTransitionsForExplicitIntent(final IntentInformation inIntentInfo) {

		if (inIntentInfo.getTargetClasses() == null || inIntentInfo.getTargetClasses().isEmpty()) {
			this.mAnalysisGraph.addMessage(new Message(MessageType.INFO, "No Target",
					"No target class could not be found for explicit intent definition"));
			return;
		}

		final EnhancedInput eInput = (EnhancedInput) this.mAnalysisGraph.getInput();
		final List<Unit> launchingStmts = inIntentInfo.getLstLaunchingUnits();
		if (launchingStmts != null && !launchingStmts.isEmpty()) {
			for (final Unit launchingStmt : launchingStmts) {
				final String strLaunchingStmt = launchingStmt.toString();
				final SootMethod callerMethod = eInput.getBodyForUnit(launchingStmt).getMethod();
				final SootClass callerClass = callerMethod.getDeclaringClass();

				Transition transition;

				for (final SootClass target : inIntentInfo.getTargetClasses()) {
					// transition from statement to target class
					transition = new TransitionLvl2a(launchingStmt, target, TransitionType.CALL);
					this.mAnalysisGraph.addTransition(transition);
				}

				if (strLaunchingStmt.contains(Constants.METHOD_START_ACTIVITY_FOR_RESULT)
						|| strLaunchingStmt.contains(Constants.METHOD_START_ACTIVITY_FOR_RESULT_BUNDLE)) {
					for (final SootClass target : inIntentInfo.getTargetClasses()) {
						/*
						 * process for case startActivityWithResult. An CALL
						 * EDGE will be added from the setResult statement to
						 * the method onReceiveResult of the caller class
						 */
						processMethodStartActivityForResult(target, callerClass, launchingStmt);
					}
				}
			}
		}
	}

	/**
	 * add transition CALL from the statement startActivityWithResult to the
	 * method onActivityResult of the same class
	 *
	 * @param inIntentInfo
	 */
	public void addTransitionsForImplicitIntent(final IntentInformation inIntentInfo) {

		if (inIntentInfo.getTargetClasses() == null || inIntentInfo.getTargetClasses().isEmpty()) {
			this.mAnalysisGraph.addMessage(new Message(MessageType.INFO, "No Target",
					"No target class could not be found for explicit intent definition"));
			return;
		}

		final EnhancedInput eInput = (EnhancedInput) this.mAnalysisGraph.getInput();
		final List<Unit> launchingStmts = inIntentInfo.getLstLaunchingUnits();

		if (launchingStmts != null && !launchingStmts.isEmpty()) {
			for (final Unit launchingStmt : launchingStmts) {
				final String strLaunchingStmt = launchingStmt.toString();
				final SootMethod callerMethod = eInput.getBodyForUnit(launchingStmt).getMethod();
				final SootClass callerClass = callerMethod.getDeclaringClass();

				if (inIntentInfo.getTargetClasses() != null && !inIntentInfo.getTargetClasses().isEmpty()) {
					Transition transition;
					for (final SootClass target : inIntentInfo.getTargetClasses()) {
						// transition from statement to target class
						transition = new TransitionLvl2a(launchingStmt, target, TransitionType.CALL);
						this.mAnalysisGraph.addTransition(transition);

					}
				}

				if (strLaunchingStmt.contains(Constants.METHOD_START_ACTIVITY_FOR_RESULT)
						|| strLaunchingStmt.contains(Constants.METHOD_START_ACTIVITY_FOR_RESULT_BUNDLE)) {
					if (inIntentInfo.getTargetClasses() != null && !inIntentInfo.getTargetClasses().isEmpty()) {
						for (final SootClass target : inIntentInfo.getTargetClasses()) {
							/*
							 * process for case startActivityWithResult. An CALL
							 * EDGE will be added from the setResult statement
							 * to the method onReceiveResult of the caller class
							 */
							processMethodStartActivityForResult(target, callerClass, launchingStmt);
						}
					} else {
						processMethodStartActivityForResult(null, callerClass, launchingStmt);
					}
				}
			}
		}
	}

	/**
	 * process for method call StartActivityForResult
	 *
	 * @param inTargetClass
	 *            the target clas of the intent
	 * @param inCallerClass
	 *            the caller class which has
	 * @param inLaunchingIntent
	 */
	private void processMethodStartActivityForResult(final SootClass inTargetClass, final SootClass inCallerClass,
			final Unit inLaunchingIntent) {

		SootMethod smOnActivityResult = null;
		List<SootMethod> lstSMs = inCallerClass.getMethods();
		if (lstSMs != null && !lstSMs.isEmpty()) {
			for (final SootMethod sm : lstSMs) {
				final String subSignature = sm.getSubSignature();
				if (subSignature.contains(Constants.METHOD_ON_ACTIVITY_RESULT)) {
					smOnActivityResult = sm;
					break;
				}
			}
		}

		if (smOnActivityResult != null) {
			if (inTargetClass != null) {
				lstSMs = inTargetClass.getMethods();
				for (final SootMethod sm : lstSMs) {
					final Body body = sm.retrieveActiveBody();
					final PatchingChain<Unit> chainUnts = body.getUnits();
					if (chainUnts != null & !chainUnts.isEmpty()) {
						for (final Unit un : chainUnts) {
							final String strUn = un.toString();
							/*
							 * in case the strUn is a setResult call statement
							 */
							if (!strUn.contains(Constants.KEY_WORD_IF) && !strUn.contains(Constants.KEY_WORD_GOTO)) {
								if (strUn.contains(Constants.METHOD_SET_RESULT1)
										|| strUn.contains(Constants.METHOD_SET_RESULT2)) {
									// add transition here
									final TransitionLvl2a transition = new TransitionLvl2a(un, smOnActivityResult,
											TransitionType.CALL);
									this.mAnalysisGraph.addTransition(transition);
								}
							}
						} // end loop of chainUnts
					} // end if of checking chainUnts valid
				} // end loop
			} else {
				final TransitionLvl2a transition = new TransitionLvl2a(inLaunchingIntent, smOnActivityResult,
						TransitionType.CALL);
				this.mAnalysisGraph.addTransition(transition);
			}

		} // end if of finding out the method onActivityResult
	}

	// computes summary edges according to the algorithm presented in
	// 'Information Flow Control for Java' by Hammer, Christian
	private void addSummaryEdges() {
		final Map<ParameterNode, ParameterNode> summaryEdges = new HashMap<>();
		final Map<Object, List<Object>> pathEdge = new HashMap<>();
		final Map<Object, List<Object>> workList = new HashMap<>();
		final Map<Object, Map<Object, List<Object>>> fragmentPath = new HashMap<>();
		// initialize worklist
		for (final SootMethod sootMethod : this.mAnalysisGraph.getParameterizedMethods()) {
			for (final ParameterNode parameter : this.mAnalysisGraph.getParameterNodes(sootMethod)) {
				if (parameter.getType() == ParamType.FORMAL_OUT) {
					pathEdge.put(parameter, new ArrayList<>());
					pathEdge.get(parameter).add(parameter);
					workList.put(parameter, new ArrayList<>());
					workList.get(parameter).add(parameter);
				}
			}
		}
		// proceed through worklist
		while (!workList.isEmpty()) {
			// get entry from worklist and delete it in worklist
			final Object from = workList.keySet().iterator().next();
			final Object to = workList.get(from).get(0);
			workList.get(from).remove(0);
			if (workList.get(from).isEmpty()) {
				workList.remove(from);
			}

			// case 1
			if (from instanceof ParameterNode && ((ParameterNode) from).getType() == ParamType.ACTUAL_OUT) {
				final ParameterNode v = (ParameterNode) from;
				for (final Object key : summaryEdges.keySet()) {
					if (summaryEdges.get(key).equals(v)) {
						propagate(pathEdge, workList, fragmentPath, key, to);
					}
				}
				for (final Transition incmTrans : this.mAnalysisGraph.getIncomingTransitions(v)) {
					if (incmTrans instanceof TransitionLvl2a) {
						final TransitionLvl2a incmTrans2a = (TransitionLvl2a) incmTrans;
						if (incmTrans2a.getTransitionType() == TransitionType.CONTROLDEPENDENCY) {
							propagate(pathEdge, workList, fragmentPath, incmTrans.getSource(), to);
						}
					}
				}

				// case 2
			} else if (from instanceof ParameterNode && ((ParameterNode) from).getType() == ParamType.FORMAL_IN) {
				// get corresponding method of 'to'-node
				SootMethod methodOfTo = null;

				if (to instanceof Unit) {
					final Unit unit = (Unit) to;
					methodOfTo = ((EnhancedInput) this.mAnalysisGraph.getInput()).getBodyForUnit(unit).getMethod();

				} else if (to instanceof SootMethod) {
					methodOfTo = (SootMethod) to;

				} else if (to instanceof ParameterNode) {
					final ParameterNode param = (ParameterNode) to;
					if (param.belongsToMethod()) {
						methodOfTo = param.getCorrespondingMethod();
					} else {
						final Unit invokeExpr = param.getCorrespondingUnit();
						final Stmt s = (Stmt) invokeExpr;
						if (s.containsInvokeExpr()) {
							methodOfTo = s.getInvokeExpr().getMethod();
						}
					}
				}

				// get callees of a method
				final Set<Unit> paramStmnts = this.mAnalysisGraph.getParameterizedInvokeStmnts();
				for (final Unit caller : paramStmnts) {
					// check whether it is a caller of the right method
					final Stmt stmt = (Stmt) caller;
					if (stmt.containsInvokeExpr()) {
						if (stmt.getInvokeExpr().getMethod().equals(methodOfTo)) {
							final ParameterNode actualIn = getCorrespondingActualIn(caller, (ParameterNode) from);
							final ParameterNode actualOut = getCorrespondingActualOut(caller, methodOfTo);
							if (actualOut != null && actualIn != null) {
								summaryEdges.put(actualIn, actualOut);
							}

							if (fragmentPath.get(to) != null && fragmentPath.get(to).get(to) != null) {
								for (final Object a : fragmentPath.get(to).get(to)) {
									propagate(pathEdge, workList, fragmentPath, from, a);
								}
							}
						}
					}
				}

				// default case
			} else {

				for (final Transition incmTrans : this.mAnalysisGraph.getIncomingTransitions(from)) {
					if (incmTrans instanceof TransitionLvl2a) {
						final TransitionLvl2a incmTrans2a = (TransitionLvl2a) incmTrans;
						if (incmTrans2a.getTransitionType() == TransitionType.DATAFLOW) {
							propagate(pathEdge, workList, fragmentPath, incmTrans.getSource(), to);
						} else if (incmTrans2a.getTransitionType() == TransitionType.CONTROLDEPENDENCY) {
							if (!(incmTrans.getSource() instanceof ParameterNode
									&& incmTrans.getTarget() instanceof ParameterNode)) {
								propagate(pathEdge, workList, fragmentPath, incmTrans.getSource(), to);
							}
						}

					}
				}
			}
		}

		// create summary transition for every entry in the list 'summaryEdges'
		for (final Entry<ParameterNode, ParameterNode> entry : summaryEdges.entrySet()) {
			final TransitionLvl2a summaryTrans = createTransition(entry.getKey(), entry.getValue(),
					TransitionType.SUMMARY);
			this.mAnalysisGraph.addTransition(summaryTrans);
		}
	}

	/**
	 * finds the corresponding actual in node for the formal in node of an
	 * invoke statememt
	 *
	 * @param caller
	 *            the invoke statement
	 * @param formalIn
	 *            the formal in node for which the corresponding actual in node
	 *            has to be found
	 * @return the corresponding actual in parameter node
	 */
	private ParameterNode getCorrespondingActualIn(final Unit caller, final ParameterNode formalIn) {

		final Stmt stmt = (Stmt) caller;
		final Set<ParameterNode> paramNodes = this.mAnalysisGraph.getParameterNodes(stmt);

		for (final ParameterNode pn : paramNodes) {
			if (pn.getIndex() == formalIn.getIndex() && pn.getType() == ParamType.ACTUAL_IN) {
				return pn;
			}
		}
		return null;
	}

	/**
	 * finds the corresponding actual out node of an invoke statememt
	 *
	 * @param caller
	 *            the invoke statement
	 * @param method
	 *            the method for which the corresponding actual out node has to
	 *            be found
	 * @return the corresponding actual out parameter node
	 */
	private ParameterNode getCorrespondingActualOut(final Unit caller, final SootMethod method) {
		for (final ParameterNode pn : this.mAnalysisGraph.getParameterNodes(caller)) {
			if (pn.getType() == ParamType.ACTUAL_OUT) {
				if (pn.belongsToMethod()) {
					if (pn.getCorrespondingMethod().equals(method)) {
						return pn;
					}
				} else {
					final Unit invokeExpr = pn.getCorrespondingUnit();
					final Stmt s = (Stmt) invokeExpr;
					if (s.containsInvokeExpr()) {
						if (s.getInvokeExpr().getMethod().equals(method)) {
							return pn;
						}
					}
				}
			}
		}
		return null;
	}

	/**
	 * used for computation of summary edges
	 */
	private void propagate(final Map<Object, List<Object>> pathEdge, final Map<Object, List<Object>> workList,
			final Map<Object, Map<Object, List<Object>>> fragmentPath, final Object from, final Object to) {
		if (pathEdge.get(from) == null) {
			pathEdge.put(from, new ArrayList<>());
		}

		if (!pathEdge.get(from).contains(to)) {
			pathEdge.get(from).add(to);
			if (workList.get(from) == null) {
				workList.put(from, new ArrayList<>());
			}
			workList.get(from).add(to);
			if (from instanceof ParameterNode && ((ParameterNode) from).getType() == ParamType.ACTUAL_OUT) {
				if (fragmentPath.get(to) == null) {
					fragmentPath.put(to, new HashMap<Object, List<Object>>());
				}
				if (fragmentPath.get(to).get(from) == null) {
					fragmentPath.get(to).put(from, new ArrayList<>());

				}
				fragmentPath.get(to).get(from).add(to);
			}

		}
	}

	private boolean checkCallBackMethod(final SootMethod inSootMethod) {
		if (inSootMethod != null) {
			final String subSig = inSootMethod.getSubSignature();
			/*
			 * ---- ACTIVITY -----------------------------
			 */
			if (Constants.getActivityLifecycleMethods().contains(subSig)) {
				return true;
			}
			/*
			 * ---- SERVICE ------------------------------
			 */
			if (Constants.getServiceLifecycleMethods().contains(subSig)) {
				return true;
			}
			/*
			 * ---- BROADCAST RECEIVER -------------------
			 */
			if (Constants.getBroadcastLifecycleMethods().contains(subSig)) {
				return true;
			}
			/*
			 * ---- CONTENT PROVIDER ---------------------
			 */
			if (Constants.getContentproviderLifecycleMethods().contains(subSig)) {
				return true;
			}
		}
		return false;
	}

	private void addMethodCallEdges() {

		final SootClass[] classes = ((EnhancedInput) this.mAnalysisGraph.getInput()).getAppClasses();
		TransitionLvl2a callTrans;

		final Collection<SootClass> colAndroidComponents = ((EnhancedInput) this.mAnalysisGraph.getInput())
				.getAndroidComponents();

		for (final SootClass sc : classes) {
			final List<SootMethod> methods = sc.getMethods();

			if (colAndroidComponents != null && colAndroidComponents.contains(sc)) {
				// in case method is callback method in lifecycle
				for (final SootMethod sm : methods) {
					if (checkCallBackMethod(sm)) {
						callTrans = new TransitionLvl2a(sc, sm, TransitionType.CALL);
						this.mAnalysisGraph.addTransition(callTrans);
					}
				}
			}

			for (final SootMethod sm : methods) {

				if (sm.hasActiveBody()) {
					final PatchingChain<Unit> units = sm.getActiveBody().getUnits();
					for (final Unit u : units) {
						final Stmt stmt = (Stmt) u;
						if (stmt.containsInvokeExpr()) {
							final SootMethod callee = stmt.getInvokeExpr().getMethod();
							callTrans = new TransitionLvl2a(stmt, callee, TransitionType.CALL);
							this.mAnalysisGraph.addTransition(callTrans);
						}
					}
				} else {
					LOGGER.trace("Found Method " + sm.getName() + " without active body");
				}
			}
		}
	}

	/**
	 * Method recursively adds Data edges between statements associated with an
	 * intent
	 */
	private void computeAndAddDataEdgesForIntents() {

		final EnhancedInput ei = (EnhancedInput) this.mAnalysisGraph.getInput();

		// Add data edges between all Units associated with intent local
		// variable
		final Map<String, Collection<IntentInformation>> intents = ei.getIntents();
		for (final String intent : intents.keySet()) {
			final Collection<IntentInformation> intentInfoCollection = intents.get(intent);
			for (final IntentInformation intentInfo : intentInfoCollection) {
				if (intentInfo.getTypeIntent() == IntentInformation.TYPE_EXPLICIT) {
					final List<Unit> inStmntList = intentInfo.getLstLaunchingUnits();

					for (final Unit inStmnt : inStmntList) {
						final Local intentLoc = getIntentLocalVariable(inStmnt);
						final HashSet<Unit> visitedNodes = new HashSet<>();
						computeDataEdgesForIntent(inStmnt, inStmnt, intentLoc, visitedNodes);
					}
				}

			}
		}

		for (final Transition newDataTransition : this.intentDataTransitions) {

			this.mAnalysisGraph.addTransition(newDataTransition);
		}

	}

	/**
	 * Method identifies the local variable associated with intent
	 *
	 * @param inUnit
	 *            statement that starts the unit
	 * @return Local variable
	 */
	private Local getIntentLocalVariable(final Unit inUnit) {

		Local loc = null;
		final List<ValueBox> vBoxes = inUnit.getUseBoxes();
		for (final ValueBox vb : vBoxes) {
			final Value v = vb.getValue();
			if (v != null && v instanceof Local) {
				final Local l = (Local) v;
				if (l.getType().toString().equals(Constants.KEY_WORD_INTENT)) {
					loc = l;
					continue;
				}
			}
		}

		return loc;
	}

	/**
	 * Adds data flow transitions between statements associated with intent
	 *
	 * @param currUnit
	 *            Current statement
	 * @param predUnit
	 *            predecessor statement
	 * @param intentLoc
	 *            local variable for the intent object
	 * @param visitedNodes
	 *            Set of nodes already visited
	 */
	private void computeDataEdgesForIntent(Unit currUnit, final Unit predUnit, final Local intentLoc,
			final HashSet<Unit> visitedNodes) {
		if (!visitedNodes.contains(predUnit)) {
			visitedNodes.add(predUnit);
			final Set<Transition> transitions = this.mAnalysisGraph.getIncomingTransitions(predUnit);
			for (final Transition transition : transitions) {
				if (transition instanceof TransitionLvl2a) {
					final TransitionLvl2a trans2a = (TransitionLvl2a) transition;
					if (trans2a.getTransitionType().equals(TransitionType.CONTROLFLOW)) {
						if (trans2a.getSource() instanceof Unit) {
							final Unit preUnit = (Unit) trans2a.getSource();
							if (checkUnitForIntentLocal(preUnit, intentLoc)) {
								final TransitionLvl2a newDataTransition = new TransitionLvl2a(preUnit, currUnit,
										TransitionType.DATAFLOW);
								this.intentDataTransitions.add(newDataTransition);

								currUnit = preUnit;
								computeDataEdgesForIntent(currUnit, preUnit, intentLoc, visitedNodes);
							} else {
								computeDataEdgesForIntent(currUnit, preUnit, intentLoc, visitedNodes);
							}
						} else {
							continue;
						}
					}
				}
			}
		}

	}

	/**
	 * Check if the statement uses intent variable
	 *
	 * @param preUnit
	 *            Statement
	 * @param intentLoc
	 *            Variable for the intent object
	 * @return True if intent local is uses in the statement, false otherwise
	 */
	private boolean checkUnitForIntentLocal(final Unit preUnit, final Local intentLoc) {
		final List<ValueBox> vBoxes = preUnit.getUseBoxes();
		final List<Local> unitLocals = new ArrayList<>();
		for (final ValueBox vBox : vBoxes) {
			if (vBox.getValue() instanceof Local) {
				unitLocals.add((Local) vBox.getValue());
			}
		}
		if (unitLocals.contains(intentLoc)) {
			return true;
		}

		return false;
	}
}
