package de.upb.pga3.panda2.extension.lvl2b;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.upb.pga3.panda2.core.datastructures.AnalysisResult;
import de.upb.pga3.panda2.core.datastructures.DetailLevel;
import de.upb.pga3.panda2.core.datastructures.Permission;
import de.upb.pga3.panda2.utilities.Constants;
import de.upb.pga3.panda2.utilities.HTMLFrameBuilder;

/**
 * This is the datastructure used for any Level 2b analysis result. To do so, it
 * is extending the class {@link AnalysisResult}.
 *
 * @author Felix
 *
 */
public class AnalysisResultLvl2b extends AnalysisResult {
	private static final Logger LOGGER = LogManager.getLogger(AnalysisResultLvl2b.class);

	// AnalysisResultLvl2b datastructures
	private ResultLeafLvl2b app;
	private ResultTreeLvl2b components;

	// Previous result
	private AnalysisResultLvl2b prevResult;
	private Map<ResultLeafLvl2b, ResultLeafLvl2b> mapCurrentToPrev;

	// List of filters
	List<String> filters;
	List<String> activeFilters;

	// permission uses
	private int[] permissionUses;
	private int[] permissionUsesAll = null;
	private int newPermissions, changedPermissions, unchangedPermissions;

	// Graphic settings
	final int height = 1000;
	public static int barWidth = 75;
	final int permissionsStart = 250;
	final int permissionsWidth = 275;
	final int barsFirst = 0;
	final int width = 2 * this.permissionsStart + this.permissionsWidth;
	final int barsSecond = this.width - AnalysisResultLvl2b.barWidth;

	// Map of permissions
	Map<Permission, SVGPermissionLvl2b> mapOfPermissions = new HashMap<>();

	// Files
	private File htmlFileTextual;
	private File htmlFileGraphical;

	// Tree option
	public static final String TREEOPTION = "Expandable tree";

	public AnalysisResultLvl2b() {
		setResultIsFile(true);

		// Files
		this.htmlFileTextual = new File("data/temp/Level2b_Result_textual.html");
		this.htmlFileGraphical = new File("data/temp/Level2b_Result_graphical.html");

		// Initially mode is summary
		this.prevResult = null;

		// Initializing filterlist
		this.filters = new ArrayList<>();
		this.filters.add(AnalysisResultLvl2b.TREEOPTION);
		this.filters.add(ResultTypeLvl2b.getStringOfType(ResultTypeLvl2b.REQUIRED));
		this.filters.add(ResultTypeLvl2b.getStringOfType(ResultTypeLvl2b.MAYBE_REQUIRED));
		this.filters.add(ResultTypeLvl2b.getStringOfType(ResultTypeLvl2b.UNUSED));
		this.filters.add(ResultTypeLvl2b.getStringOfType(ResultTypeLvl2b.MISSING));
		this.filters.add(ResultTypeLvl2b.getStringOfType(ResultTypeLvl2b.MAYBE_MISSING));
	}

	/**
	 * The next 5 methods are simple setter methods for parts of the ovverall
	 * result. The parts can be differentiated by the detail level and the group
	 * they belong to.
	 */
	public void setApp(final ResultLeafLvl2b result) {
		this.app = result;
		updateFilters(result);
	}

	public ResultLeafLvl2b getApp() {
		return this.app;
	}

	public void setComponents(final ResultTreeLvl2b result) {
		this.components = result;
		updateFilters(result);
	}

	public ResultTreeLvl2b getComponents() {
		return this.components;
	}

	public int[] getPermissionUses(final DetailLevel inDetailLvl, final List<String> inFilters) {
		getTextualResult(inDetailLvl, inFilters, true);
		return this.permissionUses;
	}

	private void updateFilters(final ResultTreeLvl2b result) {
		if (result != null) {
			for (final ResultLeafLvl2b leaf : result.getLeafs().values()) {
				updateFilters(leaf);
			}
		}
	}

	private void updateFilters(final ResultLeafLvl2b result) {
		if (result != null && !result.getResultItems(ResultTypeLvl2b.ALL).isEmpty()) {
			for (final ResultItemLvl2b resultItem : result.getResultItems(ResultTypeLvl2b.ALL)) {
				boolean add = true;
				for (final String filter : this.filters) {
					if (filter.equals(Constants.PREFIX_PERMISSION + " " + resultItem.getPermission().getName())) {
						add = false;
						break;
					}
				}
				if (add) {
					this.filters.add(Constants.PREFIX_PERMISSION + " " + resultItem.getPermission().getName());
				}
			}
		}
	}

	/**
	 * This method will be called once a comparison result is created. It will
	 * bind elements from the analyzed App to elements from the previously
	 * analyzed App.
	 *
	 * @param prevRes
	 *            This parameter contains the previously computed result.
	 */
	public void compare(final AnalysisResult prevRes) {
		// Setting mode to comparison
		this.prevResult = (AnalysisResultLvl2b) prevRes;
		this.mapOfPermissions = this.prevResult.getMapOfPermissions();
		this.mapCurrentToPrev = new HashMap<>();

		// App
		this.mapCurrentToPrev.put(this.app, this.prevResult.getApp());

		// Components
		for (final ResultLeafLvl2b currentLeaf : this.components.getLeafsAsList()) {
			for (final ResultLeafLvl2b prevLeaf : this.prevResult.getComponents().getLeafsAsList()) {
				if (currentLeaf.getName().equals(prevLeaf.getName())) {
					this.mapCurrentToPrev.put(currentLeaf, prevLeaf);
				}
			}
		}
	}

	@Override
	public String getGraphicalResult(final DetailLevel inDetailLvl, final List<String> inFilters,
			final boolean inShowStats) {
		try {
			// Reset output file
			resetFile(this.htmlFileGraphical);

			// Setup
			final BufferedWriter bw = new BufferedWriter(new FileWriter(this.htmlFileGraphical));
			final HTMLFrameBuilder fb = new HTMLFrameBuilder(bw,
					"Graphical result - Inter-App Permission Usage Analysis");
			fb.setHeaderAutoHide(inShowStats);
			fb.setHintPersistent(false);
			fb.setCustomStyle(
					"div.icon { float: left; border: 1px solid #000000; width: 12px; height: 7px;	margin-right: 5px; margin-top: 3px;	}");

			// Legend
			fb.addLegendEntry("<div class=\"icon\" style=\"background:#007f22;\"></div>", "REQUIRED");
			fb.addLegendEntry("<div class=\"icon\" style=\"background:#0006fd;\"></div>", "MAYBE REQUIRED");
			fb.addLegendEntry("<div class=\"icon\" style=\"background:#927a3a;\"></div>", "UNUSED");
			fb.addLegendEntry("<div class=\"icon\" style=\"background:#ff7500;\"></div>", "MAYBE MISSING");
			fb.addLegendEntry("<div class=\"icon\" style=\"background:#ff0000;\"></div>", "MISSING");
			fb.addLegendEntry("<br />", "<br />");
			fb.addLegendEntry(
					"<div class=\"icon\" style=\"background:#000000; height:1px; width:50px; border:none;\"></div>",
					"Direct");
			fb.addLegendEntry(
					"<div class=\"icon\" style=\"background:#000000; height:1px; width:14px; border:none;\"></div><div class=\"icon\" style=\"background:#000000; height:1px; width:14px; border:none;\"></div><div class=\"icon\" style=\"background:#000000; height:1px; width:12px; border:none;\"></div>",
					"Indirect");
			fb.addLegendEntry(
					"<div class=\"icon\" style=\"background:#000000; height:3px; width:50px; border:none;\"></div>",
					"Direct &amp; Indirect");

			// Content
			fb.append("<center>");
			if (inDetailLvl == DetailLevelLvl2b.APP) {
				if (this.prevResult != null) {
					getGraphic(this.app, this.prevResult.getApp(), fb);
				} else {
					getGraphic(this.app, null, fb);
				}
			} else if (inDetailLvl == DetailLevelLvl2b.COMPONENT) {
				for (final ResultLeafLvl2b leaf : this.components.getLeafsAsList()) {
					if (leaf != null) {
						if (this.prevResult != null) {
							final ResultLeafLvl2b compareLeaf = this.prevResult.getComponents().getLeaf(leaf.getName());
							if (compareLeaf != null) {
								getGraphic(leaf, compareLeaf, fb);
							} else {
								getGraphic(leaf, null, fb);
							}
						} else {
							getGraphic(leaf, null, fb);
						}
					}
				}
			} else {
				fb.append("Graphical result only available in APP and COMPONENT mode. Please choose one of those.");
			}
			fb.append("</center>");

			// Statistics
			final String statsLine = "<strong>"
					+ (this.permissionUsesAll[ResultTypeLvl2b.REQUIRED]
							+ this.permissionUsesAll[ResultTypeLvl2b.MAYBE_REQUIRED]
							+ this.permissionUsesAll[ResultTypeLvl2b.UNUSED]
							+ this.permissionUsesAll[ResultTypeLvl2b.MAYBE_MISSING]
							+ this.permissionUsesAll[ResultTypeLvl2b.MISSING])
					+ "</strong> (<font color=\"" + ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.REQUIRED) + "\">"
					+ this.permissionUsesAll[ResultTypeLvl2b.REQUIRED] + "</font> / <font color=\""
					+ ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.MAYBE_REQUIRED) + "\">"
					+ this.permissionUsesAll[ResultTypeLvl2b.MAYBE_REQUIRED] + "</font> / <font color=\""
					+ ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.UNUSED) + "\">"
					+ this.permissionUsesAll[ResultTypeLvl2b.UNUSED] + "</font> / <font color=\""
					+ ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.MAYBE_MISSING) + "\">"
					+ this.permissionUsesAll[ResultTypeLvl2b.MAYBE_MISSING] + "</font> / <font color=\""
					+ ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.MISSING) + "\">"
					+ this.permissionUsesAll[ResultTypeLvl2b.MISSING] + "</font>)";
			fb.addStaticticsRow("Analysed permission uses:", statsLine);
			fb.setAppTrustworthy(this.permissionUsesAll[ResultTypeLvl2b.MISSING]
					+ this.permissionUsesAll[ResultTypeLvl2b.MAYBE_MISSING] == 0);
			fb.complete();
			bw.close();
		} catch (final IOException e) {
			LOGGER.error("Could not access outputfile.");
		}

		return this.htmlFileGraphical.toString();
	}

	private void getGraphic(final ResultLeafLvl2b leaf, final ResultLeafLvl2b compLeaf, final HTMLFrameBuilder fb)
			throws IOException {
		// Reset permission map
		this.mapOfPermissions.clear();

		// Generate SVG items
		final SVGGraphicLvl2b svgGraphic = getGraphic(leaf, compLeaf, this.permissionsStart, this.barsFirst,
				this.permissionsStart + this.permissionsWidth, this.barsSecond);

		// Build
		if (this.prevResult != null) {
			fb.append("<table width=\"" + this.width + "\"><tr><td>Current Result</td><td align=\"center\"><strong>"
					+ leaf.getName() + "</strong></td><td align=\"right\">Previous Result</td></tr></table>\n");
		} else {
			fb.append("<strong>" + leaf.getName() + "</strong><br />\n");
		}
		fb.append("<svg xmlns=\"http://www.w3.org/2000/svg\" width=\"" + this.width + "\" height=\""
				+ (13 + 20 * this.mapOfPermissions.values().size() + 2)
				+ "\" xmlns:xlink=\"http://www.w3.org/1999/xlink\">");
		fb.append(svgGraphic.toString());
		fb.append("</svg><br /><br />");
	}

	public SVGGraphicLvl2b getGraphic(final ResultLeafLvl2b leaf, final ResultLeafLvl2b compLeaf, final int xLineStart,
			final int xLineEnd, final int xLineStartComp, final int xLineEndComp) {
		final int[] permissionUsesGraphical = new int[5];
		final int[] permissionUsesGraphicalComp = new int[5];
		final List<SVGArrowLvl2b> arrows = new ArrayList<>();
		final SVGBarLvl2b[] bars = new SVGBarLvl2b[5];
		int yBars = 0;
		final SVGBarLvl2b[] barsComp = new SVGBarLvl2b[5];
		int yBarsComp = 0;
		int yPermissions = 13;

		for (int i = 0; i < 5; i++) {
			// Current result
			if (leaf.getResultItems(i) != null) {
				for (final ResultItemLvl2b resultItem : leaf.getResultItems(i)) {
					if (this.activeFilters.contains(ResultTypeLvl2b.getStringOfType(i)) || this.activeFilters
							.contains(Constants.PREFIX_PERMISSION + " " + resultItem.getPermission().getName())) {
						permissionUsesGraphical[i]++;

						if (!this.mapOfPermissions.containsKey(resultItem.getPermission())) {
							this.mapOfPermissions.put(resultItem.getPermission(),
									new SVGPermissionLvl2b(resultItem.getPermission(), yPermissions));
							yPermissions += 20;
						}
					}
				}
			}

			// Previous result
			if (compLeaf != null) {
				if (compLeaf.getResultItems(i) != null) {
					for (final ResultItemLvl2b resultItem : compLeaf.getResultItems(i)) {
						if (this.activeFilters.contains(ResultTypeLvl2b.getStringOfType(i)) || this.activeFilters
								.contains(Constants.PREFIX_PERMISSION + " " + resultItem.getPermission().getName())) {
							permissionUsesGraphicalComp[i]++;

							if (!this.mapOfPermissions.containsKey(resultItem.getPermission())) {
								this.mapOfPermissions.put(resultItem.getPermission(),
										new SVGPermissionLvl2b(resultItem.getPermission(), yPermissions));
								yPermissions += 20;
							}
						}
					}
				}
			}
		}

		final int sumPermissions = permissionUsesGraphical[ResultTypeLvl2b.REQUIRED]
				+ permissionUsesGraphical[ResultTypeLvl2b.MAYBE_REQUIRED]
				+ permissionUsesGraphical[ResultTypeLvl2b.UNUSED]
				+ permissionUsesGraphical[ResultTypeLvl2b.MAYBE_MISSING]
				+ permissionUsesGraphical[ResultTypeLvl2b.MISSING];

		for (int i = 0; i < 5; i++) {
			final int barHeight = Math
					.round((float) permissionUsesGraphical[i] / sumPermissions * (13 + 20 * sumPermissions));
			bars[i] = new SVGBarLvl2b(xLineEnd, yBars, barHeight);
			yBars = yBars + barHeight;
			if (compLeaf != null) {
				final int barHeightComp = Math
						.round((float) permissionUsesGraphicalComp[i] / sumPermissions * (13 + 20 * sumPermissions));
				barsComp[i] = new SVGBarLvl2b(xLineEndComp, yBarsComp, barHeightComp);
				yBarsComp = yBarsComp + barHeightComp;
			}

			// Current result
			if (leaf.getResultItems(i) != null) {
				for (final ResultItemLvl2b resultItem : leaf.getResultItems(i)) {
					if (this.activeFilters.contains(ResultTypeLvl2b.getStringOfType(i)) || this.activeFilters
							.contains(Constants.PREFIX_PERMISSION + " " + resultItem.getPermission().getName())) {
						final int yLine = this.mapOfPermissions.get(resultItem.getPermission()).getY() - 4;
						final int[] offset = new int[] { -5, AnalysisResultLvl2b.barWidth };
						arrows.add(new SVGArrowLvl2b(new int[] { xLineStart + offset[0], yLine },
								new int[] { xLineEnd + offset[1], bars[i].getClosestTo(yLine) },
								resultItem.getDirect()));
					}
				}
			}

			// Previous result
			if (compLeaf != null) {
				if (compLeaf.getResultItems(i) != null) {
					for (final ResultItemLvl2b resultItem : compLeaf.getResultItems(i)) {
						if (this.activeFilters.contains(ResultTypeLvl2b.getStringOfType(i)) || this.activeFilters
								.contains(Constants.PREFIX_PERMISSION + " " + resultItem.getPermission().getName())) {
							final int yLine = this.mapOfPermissions.get(resultItem.getPermission()).getY() - 4;
							final int[] offset = new int[] { 5, 0 };
							arrows.add(new SVGArrowLvl2b(new int[] { xLineStartComp + offset[0], yLine },
									new int[] { xLineEndComp + offset[1], barsComp[i].getClosestTo(yLine) },
									resultItem.getDirect()));
						}
					}
				}
			}
		}

		return new SVGGraphicLvl2b(arrows, this.mapOfPermissions, bars, barsComp, xLineStart, compLeaf != null);
	}

	@Override
	public String getTextualResult(final DetailLevel inDetailLvl, final List<String> inFilters,
			final boolean inShowStats) {
		// Counters
		this.permissionUses = new int[5];
		this.newPermissions = 0;
		this.changedPermissions = 0;
		this.unchangedPermissions = 0;

		int[] prevPermissionUses = null;
		if (this.prevResult != null) {
			prevPermissionUses = this.prevResult.getPermissionUses(inDetailLvl, inFilters);
		}

		// Assign filters
		if (inFilters == null) {
			this.activeFilters = getFilters();
			this.activeFilters.remove(AnalysisResultLvl2b.TREEOPTION);
		} else {
			this.activeFilters = inFilters;
		}

		try {
			// Reset output file
			resetFile(this.htmlFileGraphical);

			// Setup
			final BufferedWriter bw = new BufferedWriter(new FileWriter(this.htmlFileTextual));
			final HTMLFrameBuilder fb = new HTMLFrameBuilder(bw,
					"Textual result - Inter-App Permission Usage Analysis");
			fb.setHeaderAutoHide(inShowStats);
			fb.setHintPersistent(false);
			fb.setCustomStyle(
					"div.icon { float: left; border: 1px solid #000000; width: 12px; height: 7px;	margin-right: 5px; margin-top: 3px;	} div.plus { background: #C9C9C9; float: left; border: 1px outset #CCCCCC; padding: 0px 0px 0px 0px; margin-top: 2px; margin-right: 10px; height: 9px; width: 9px; } div.inner { margin-top:-2px; font-size: 10px; text-align: center; cursor: pointer;	} div.spacer { float: left;	margin-top: 3px; margin-right: 10px; height: 11px; width: 11px;	} div.popout { padding-left: 25px; visibility: hidden; max-height: 0px;	overflow: hidden; }	div.premissions { padding-left: 21px; }	table {	border-spacing:0px;	} td { padding: 0px; }");
			fb.setCustomScript(
					"function show(caller) {	if(caller.nextElementSibling.nextElementSibling.nextElementSibling.nextElementSibling.nextElementSibling.style.visibility == 'visible') { caller.nextElementSibling.nextElementSibling.nextElementSibling.nextElementSibling.nextElementSibling.style.maxHeight='0px'; caller.nextElementSibling.nextElementSibling.nextElementSibling.nextElementSibling.nextElementSibling.style.visibility='hidden';	caller.firstElementChild.innerHTML='+';	caller.style.border='1px outset #CCCCCC'; } else { caller.nextElementSibling.nextElementSibling.nextElementSibling.nextElementSibling.nextElementSibling.style.maxHeight='99999px';	caller.nextElementSibling.nextElementSibling.nextElementSibling.nextElementSibling.nextElementSibling.style.visibility='visible'; caller.firstElementChild.innerHTML='-'; caller.style.border='1px inset #CCCCCC'; }}");

			// Legend
			fb.addLegendEntry("<div class=\"icon\" style=\"background:#007f22;\"></div>", "REQUIRED");
			fb.addLegendEntry("<div class=\"icon\" style=\"background:#0006fd;\"></div>", "MAYBE REQUIRED");
			fb.addLegendEntry("<div class=\"icon\" style=\"background:#927a3a;\"></div>", "UNUSED");
			fb.addLegendEntry("<div class=\"icon\" style=\"background:#ff7500;\"></div>", "MAYBE MISSING");
			fb.addLegendEntry("<div class=\"icon\" style=\"background:#ff0000;\"></div>", "MISSING");

			// Content
			if (inDetailLvl == DetailLevelLvl2b.COMPONENT) {
				getTextualForTree(this.components, inDetailLvl, fb);
			} else {
				getTextualForLeaf(this.app, getChildren(this.app, inDetailLvl), inDetailLvl, fb);
			}

			// Statistics
			if (inDetailLvl == DetailLevelLvl2b.APP && this.permissionUsesAll == null) {
				this.permissionUsesAll = this.permissionUses;
			} else if (inDetailLvl != DetailLevelLvl2b.APP && this.permissionUsesAll == null) {
				getTextualResult(DetailLevelLvl2b.APP, inFilters, inShowStats);
			}

			String statsLine = "<strong>"
					+ (this.permissionUsesAll[ResultTypeLvl2b.REQUIRED]
							+ this.permissionUsesAll[ResultTypeLvl2b.MAYBE_REQUIRED]
							+ this.permissionUsesAll[ResultTypeLvl2b.UNUSED]
							+ this.permissionUsesAll[ResultTypeLvl2b.MAYBE_MISSING]
							+ this.permissionUsesAll[ResultTypeLvl2b.MISSING])
					+ "</strong> (<font color=\"" + ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.REQUIRED) + "\">"
					+ this.permissionUsesAll[ResultTypeLvl2b.REQUIRED] + "</font> / <font color=\""
					+ ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.MAYBE_REQUIRED) + "\">"
					+ this.permissionUsesAll[ResultTypeLvl2b.MAYBE_REQUIRED] + "</font> / <font color=\""
					+ ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.UNUSED) + "\">"
					+ this.permissionUsesAll[ResultTypeLvl2b.UNUSED] + "</font> / <font color=\""
					+ ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.MAYBE_MISSING) + "\">"
					+ this.permissionUsesAll[ResultTypeLvl2b.MAYBE_MISSING] + "</font> / <font color=\""
					+ ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.MISSING) + "\">"
					+ this.permissionUsesAll[ResultTypeLvl2b.MISSING] + "</font>)";
			fb.addStaticticsRow("Analysed permission uses:", statsLine);
			fb.addStaticticsRow("<br />", "<br />");
			fb.addStaticticsRow("<strong>Filtered</strong>", "");
			if (this.prevResult != null) {
				statsLine = "<strong>" + this.newPermissions + "</strong> / <strong>" + this.unchangedPermissions
						+ "</strong> / <strong>" + this.changedPermissions + "</strong>";
				fb.addStaticticsRow("New/Un-/Changed permissions:", statsLine);

				statsLine = "<strong>"
						+ (prevPermissionUses[ResultTypeLvl2b.REQUIRED]
								+ prevPermissionUses[ResultTypeLvl2b.MAYBE_REQUIRED]
								+ prevPermissionUses[ResultTypeLvl2b.UNUSED]
								+ prevPermissionUses[ResultTypeLvl2b.MAYBE_MISSING]
								+ prevPermissionUses[ResultTypeLvl2b.MISSING])
						+ "</strong> (<font color=\"" + ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.REQUIRED) + "\">"
						+ prevPermissionUses[ResultTypeLvl2b.REQUIRED] + "</font> / <font color=\""
						+ ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.MAYBE_REQUIRED) + "\">"
						+ prevPermissionUses[ResultTypeLvl2b.MAYBE_REQUIRED] + "</font> / <font color=\""
						+ ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.UNUSED) + "\">"
						+ prevPermissionUses[ResultTypeLvl2b.UNUSED] + "</font> / <font color=\""
						+ ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.MAYBE_MISSING) + "\">"
						+ prevPermissionUses[ResultTypeLvl2b.MAYBE_MISSING] + "</font> / <font color=\""
						+ ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.MISSING) + "\">"
						+ prevPermissionUses[ResultTypeLvl2b.MISSING] + "</font>) &#8594; ";
			} else {
				statsLine = "";
			}
			statsLine += "<strong>" + (this.permissionUses[ResultTypeLvl2b.REQUIRED]
					+ this.permissionUses[ResultTypeLvl2b.MAYBE_REQUIRED] + this.permissionUses[ResultTypeLvl2b.UNUSED]
					+ this.permissionUses[ResultTypeLvl2b.MAYBE_MISSING] + this.permissionUses[ResultTypeLvl2b.MISSING])
					+ "</strong> (<font color=\"" + ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.REQUIRED) + "\">"
					+ this.permissionUses[ResultTypeLvl2b.REQUIRED] + "</font> / <font color=\""
					+ ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.MAYBE_REQUIRED) + "\">"
					+ this.permissionUses[ResultTypeLvl2b.MAYBE_REQUIRED] + "</font> / <font color=\""
					+ ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.UNUSED) + "\">"
					+ this.permissionUses[ResultTypeLvl2b.UNUSED] + "</font> / <font color=\""
					+ ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.MAYBE_MISSING) + "\">"
					+ this.permissionUses[ResultTypeLvl2b.MAYBE_MISSING] + "</font> / <font color=\""
					+ ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.MISSING) + "\">"
					+ this.permissionUses[ResultTypeLvl2b.MISSING] + "</font>)";
			fb.addStaticticsRow("Analysed permission uses:", statsLine);
			if (this.prevResult != null) {
				statsLine = "<strong>"
						+ (prevPermissionUses[ResultTypeLvl2b.MAYBE_MISSING]
								+ prevPermissionUses[ResultTypeLvl2b.MISSING])
						+ "</strong> (<font color=\"" + ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.MAYBE_MISSING)
						+ "\">" + prevPermissionUses[ResultTypeLvl2b.MAYBE_MISSING] + "</font> / <font color=\""
						+ ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.MISSING) + "\">"
						+ prevPermissionUses[ResultTypeLvl2b.MISSING] + "</font>) &#8594; ";
			} else {
				statsLine = "";
			}
			statsLine += "<strong>"
					+ (this.permissionUses[ResultTypeLvl2b.MAYBE_MISSING]
							+ this.permissionUses[ResultTypeLvl2b.MISSING])
					+ "</strong> (<font color=\"" + ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.MAYBE_MISSING)
					+ "\">" + this.permissionUses[ResultTypeLvl2b.MAYBE_MISSING] + "</font> / <font color=\""
					+ ResultTypeLvl2b.getColorOfType(ResultTypeLvl2b.MISSING) + "\">"
					+ this.permissionUses[ResultTypeLvl2b.MISSING] + "</font>)";
			fb.addStaticticsRow("Possibly malicious:", statsLine);
			fb.setAppTrustworthy(this.permissionUsesAll[ResultTypeLvl2b.MISSING]
					+ this.permissionUsesAll[ResultTypeLvl2b.MAYBE_MISSING] == 0);
			fb.complete();
			bw.close();
		} catch (final IOException e) {
			LOGGER.error("Could not access outputfile.");
		}

		return this.htmlFileTextual.toString();
	}

	private ResultTreeLvl2b getChildren(final Object parent, final DetailLevel inDetailLvl) {
		if (parent == this.app) {
			if (inDetailLvl == DetailLevelLvl2b.APP || inDetailLvl == null) {
				return this.components;
			}
		}
		return null;
	}

	private void getTextualForTree(final ResultTreeLvl2b tree, final DetailLevel inDetailLvl, final HTMLFrameBuilder fb)
			throws IOException {
		if (tree != null) {
			for (final ResultLeafLvl2b childLeaf : tree.getLeafsAsList()) {
				getTextualForLeaf(childLeaf, getChildren(tree, inDetailLvl), inDetailLvl, fb);
			}
		}
	}

	private void getTextualForLeaf(final ResultLeafLvl2b leaf, final ResultTreeLvl2b children,
			final DetailLevel inDetailLvl, final HTMLFrameBuilder fb) throws IOException {
		if (children != null && this.activeFilters.contains(AnalysisResultLvl2b.TREEOPTION)) {
			fb.append("<div class=\"plus\" onClick=\"show(this)\"><div class=\"inner\">+</div></div>");
		} else {
			fb.append("<div class=\"spacer\"></div>");
		}
		fb.append(" <strong>" + leaf.getName() + "</strong><br />\n" + "<div class=\"premissions\">\n");
		final StringBuilder resultStr = new StringBuilder();
		for (int i = 0; i < 5; i++) {
			if (leaf.getResultItems(i) != null) {
				for (final ResultItemLvl2b resultItem : leaf.getResultItems(i)) {
					resultStr.append(getTextualForResultItem(resultItem, i, leaf));
				}
			}
		}
		fb.append(resultStr.toString());
		fb.append("</div>\n" + "<br />\n");
		if (this.activeFilters.contains(AnalysisResultLvl2b.TREEOPTION)) {
			fb.append("<div class=\"popout\">\n" + "\t");
			getTextualForTree(children, inDetailLvl, fb);
			fb.append("\n</div>\n\n");
		}
	}

	private String getTextualForResultItem(final ResultItemLvl2b resultItem, final int type,
			final ResultLeafLvl2b leaf) {
		if (this.activeFilters.contains(ResultTypeLvl2b.getStringOfType(type)) || this.activeFilters
				.contains(Constants.PREFIX_PERMISSION + " " + resultItem.getPermission().getName())) {
			this.permissionUses[type]++;
			if (this.prevResult == null) {
				return "\t<font color=\"" + ResultTypeLvl2b.getColorOfType(type) + "\" title=\""
						+ ResultTypeLvl2b.getStringOfType(type) + "\">" + resultItem.getPermission().getName()
						+ "</font>" + resultItem.getDirectString() + "<br />\n";
			} else {
				final ResultLeafLvl2b prevLeaf = this.mapCurrentToPrev.get(leaf);

				if (prevLeaf != null) {
					int found = -1;
					for (int i = 0; i < 5; i++) {
						if (prevLeaf.getResultItems(i) != null) {
							for (final ResultItemLvl2b prevResultItem : prevLeaf.getResultItems(i)) {
								if (prevResultItem.getPermission().getName()
										.equals(resultItem.getPermission().getName())) {
									found = i;
									break;
								}
							}
							if (found >= 0) {
								break;
							}
						}
					}
					if (type != found) {
						this.changedPermissions++;
						return "\t<strong>" + resultItem.getPermission().getName() + "</strong> <font color=\""
								+ ResultTypeLvl2b.getColorOfType(found) + "\" title=\""
								+ ResultTypeLvl2b.getStringOfType(found) + "\">"
								+ ResultTypeLvl2b.getStringOfType(found) + "</font>" + resultItem.getDirectString()
								+ " &#8594; <font color=\"" + ResultTypeLvl2b.getColorOfType(type) + "\" title=\""
								+ ResultTypeLvl2b.getStringOfType(type) + "\">" + ResultTypeLvl2b.getStringOfType(type)
								+ "</font>" + resultItem.getDirectString() + "<br />\n";
					} else {
						this.unchangedPermissions++;
						return "\t<font color=\"" + ResultTypeLvl2b.getColorOfType(type) + "\" title=\""
								+ ResultTypeLvl2b.getStringOfType(type) + "\">" + resultItem.getPermission().getName()
								+ "</font>" + resultItem.getDirectString() + " <strong>(unchanged)</strong><br />\n";
					}
				} else {
					this.newPermissions++;
					return "\t<font color=\"" + ResultTypeLvl2b.getColorOfType(type) + "\" title=\""
							+ ResultTypeLvl2b.getStringOfType(type) + "\">" + resultItem.getPermission().getName()
							+ "</font>" + resultItem.getDirectString() + " <strong>(new)</strong><br />\n";
				}
			}
		} else {
			return "";
		}
	}

	@Override
	public List<DetailLevel> getDetailLevels() {
		final List<DetailLevel> detaillevelList = new ArrayList<>();
		for (final DetailLevelLvl2b item : DetailLevelLvl2b.values()) {
			detaillevelList.add(item);
		}
		return detaillevelList;
	}

	/**
	 * Replies the map of permissions used in the graphical result.
	 *
	 * @return A map of permissions which maps permissions to graphical objects
	 *         of the graphical result.
	 */
	public Map<Permission, SVGPermissionLvl2b> getMapOfPermissions() {
		return this.mapOfPermissions;
	}

	@Override
	public List<String> getFilters() {
		return this.filters;
	}

	/**
	 * Resets an output file
	 *
	 * @param file
	 *            Defines the file to be reset.
	 * @throws IOException
	 */
	private static void resetFile(final File file) throws IOException {
		file.delete();
		file.createNewFile();
	}
}
