package de.upb.pga3.panda2.extension.lvl2b.analyzer;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.upb.pga3.panda2.core.datastructures.AnalysisGraph;
import de.upb.pga3.panda2.core.datastructures.EnhancedInput;
import de.upb.pga3.panda2.core.datastructures.Permission;
import de.upb.pga3.panda2.core.datastructures.ResultInput;
import de.upb.pga3.panda2.core.datastructures.Transition;
import de.upb.pga3.panda2.extension.lvl1.AnalysisResultLvl1;
import soot.Body;
import soot.SootClass;
import soot.SootMethod;
import soot.Unit;

/**
 * This class will transfer the permissions from an element of one App to an
 * element of another App until a fixpoint is reached and no more permissions
 * can be transfered.
 *
 * @author Felix
 */
public class FixpointComputer {
	private AnalysisGraph graph;
	private ResultInput ri;
	private ManifestPermissionComparerLvl2b mpc;
	private EnhancedInput currentEnhancedInput;
	boolean allmode;

	private final Map<EnhancedInput, List<Object>> changedElements = new HashMap<>();
	private Map<EnhancedInput, List<Transition>> worklist = new HashMap<>();

	public FixpointComputer(final AnalysisGraph graph, final ManifestPermissionComparerLvl2b mpc,
			final boolean allmode) {
		this.graph = graph;
		this.ri = (ResultInput) graph.getInput();
		this.mpc = mpc;
		this.currentEnhancedInput = (EnhancedInput) ((AnalysisResultLvl1) this.ri.getResults()
				.get(this.ri.getResults().size() - 1)).getAnalysisGraph().getInput();
		this.allmode = allmode;
	}

	/**
	 * Start the fixpoint computation.
	 */
	public void computeFixpoint() {
		initWorkList();
		while (!this.worklist.isEmpty() && !this.changedElements.isEmpty()) {
			runWorklistAlgorithm();
		}
	}

	/**
	 * Intinializes the worklist used for the fixpoint computation.
	 */
	private void initWorkList() {
		for (int i = 0; i < this.ri.getResults().size(); i++) {
			final EnhancedInput ei = (EnhancedInput) ((AnalysisResultLvl1) this.ri.getResults().get(i))
					.getAnalysisGraph().getInput();

			for (final SootClass componentOrClass : ei.getAppClasses()) {
				addElement(ei, componentOrClass);
				for (final SootMethod method : componentOrClass.getMethods()) {
					if (!componentOrClass.isInterface()) {
						try {
							addElement(ei, method);
							final Body body = method.retrieveActiveBody();
							for (final Unit stm : body.getUnits()) {
								addElement(ei, stm);
							}
						} catch (final RuntimeException e) {
							if (e.getMessage().contains("No method source set for method")) {
								// Message was already given by the Enhancer. Do
								// nothing.
							} else {
								e.printStackTrace();
							}
						}
					}
				}
			}
		}
	}

	/**
	 * Run the worklist algorithm once on the current worklist.
	 */
	private void runWorklistAlgorithm() {
		// Step 1
		for (int i = 0; i < this.ri.getResults().size(); i++) {
			final EnhancedInput ei = (EnhancedInput) ((AnalysisResultLvl1) this.ri.getResults().get(i))
					.getAnalysisGraph().getInput();
			if (this.changedElements.get(ei) != null && !this.changedElements.get(ei).isEmpty()) {
				for (final Object element : this.changedElements.get(ei)) {
					fillUpPermission(element, ei);
				}
				this.changedElements.remove(ei);
			}
		}

		// Step 2
		for (int i = 0; i < this.ri.getResults().size(); i++) {
			final EnhancedInput ei = (EnhancedInput) ((AnalysisResultLvl1) this.ri.getResults().get(i))
					.getAnalysisGraph().getInput();
			if (this.worklist.get(ei) != null && !this.worklist.get(ei).isEmpty()) {
				final List<Transition> worklistCopy = new ArrayList<>();
				worklistCopy.addAll(this.worklist.get(ei));
				this.worklist.remove(ei);

				for (final Transition transition : worklistCopy) {
					checkTransition(transition, ei);
				}
			}
		}
	}

	/**
	 * Fills the graph with permissions accordingly to the permissions assigned
	 * to the element.
	 *
	 * @param element
	 *            Element that represents the starting point for filling up the
	 *            graph with {@link Permission}s.
	 * @param ei
	 *            The {@link EnhancedInput} the element belongs to.
	 */
	private void fillUpPermission(final Object element, final EnhancedInput ei) {
		if (element instanceof Unit) {
			if (ei.getPermissionsFor((Unit) element) != null && !ei.getPermissionsFor((Unit) element).isEmpty()) {
				for (final Permission permission : ei.getPermissionsFor((Unit) element)) {
					final SootMethod method = ei.getBodyForUnit((Unit) element).getMethod();
					if (!ei.getPermissionsFor(method).contains(permission)) {
						this.mpc.addIndirectPermission(method, permission, ei);
						fillUpPermission(method, ei);
					}
				}
			}
		} else if (element instanceof SootMethod) {
			if (ei.getPermissionsFor((SootMethod) element) != null
					&& !ei.getPermissionsFor((SootMethod) element).isEmpty()) {
				for (final Permission permission : ei.getPermissionsFor((SootMethod) element)) {
					final SootClass componentOrClass = ((SootMethod) element).getDeclaringClass();
					if (!ei.getPermissionsFor(componentOrClass).contains(permission)) {
						this.mpc.addIndirectPermission(componentOrClass, permission, ei);
					}
				}
			}
		}
	}

	/**
	 * Add an element to the worklist.
	 *
	 * @param ei
	 *            {@link EnhancedInput} that contains the element that will be
	 *            added to the worklist.
	 * @param element
	 *            This represents the element that is going to be added.
	 */
	private void addElement(final EnhancedInput ei, final Object element) {
		// Object
		List<Object> tempList1 = this.changedElements.get(ei);
		if (tempList1 == null) {
			tempList1 = new ArrayList<>();
		}
		tempList1.add(element);
		this.changedElements.put(ei, tempList1);

		// Transition
		if (!this.graph.getOutgoingTransitions(element).isEmpty()) {
			List<Transition> tempList2 = this.worklist.get(ei);
			if (tempList2 == null) {
				tempList2 = new ArrayList<>();
			}
			tempList2.addAll(this.graph.getOutgoingTransitions(element));
			this.worklist.put(ei, tempList2);
		}
	}

	/**
	 * Check one transition and respectively transfer permissions and add
	 * elements to the worklist again.
	 *
	 * @param transition
	 * @param ei
	 */
	private void checkTransition(final Transition transition, final EnhancedInput ei) {
		if (this.allmode || ei == this.currentEnhancedInput) {
			final Object source = transition.getSource();
			final SootClass target = (SootClass) transition.getTarget();

			// Find enhancedInput of the target
			EnhancedInput ei2 = null;
			for (int i = 0; i < this.ri.getResults().size(); i++) {
				final EnhancedInput ei3 = (EnhancedInput) ((AnalysisResultLvl1) this.ri.getResults().get(i))
						.getAnalysisGraph().getInput();
				for (final SootClass componentOrClass : ei3.getAppClasses()) {
					if (target == componentOrClass) {
						ei2 = ei3;
						break;
					}
				}
				if (ei2 != null) {
					break;
				}
			}

			// Add permissions
			for (final Permission perm : ei.getPermissions(source)) {
				if (!ei2.getPermissions(target).contains(perm)) {
					if (ei2.getComponentByName(target.getName()) != null
							&& !ei2.getComponentByName(target.getName()).isInterface()) {
						for (final SootMethod targetMethod : ei2.getComponentByName(target.getName()).getMethods()) {
							this.mpc.addIndirectPermission(targetMethod, perm, ei2);
							addElement(ei2, targetMethod);
						}
					}
					this.mpc.addIndirectPermission(target, perm, ei2);
					addElement(ei2, target);
				}
			}
			for (final Permission perm : ei2.getPermissions(target)) {
				if (!ei.getPermissions(source).contains(perm)) {
					this.mpc.addIndirectPermission(source, perm, ei);
					addElement(ei, source);
				}
			}
		}
	}
}
